defstar {
	name {Crosscor}
	domain {CG96}
	desc { Estimates an crosscorrelation by averaging input samples. }
	version {@(#)CG96Crosscor.pl	1.7 12/8/92}
	author { Chih-Tsung Huang, ported from gabriel }
        acknowledge { Gabriel version by Anthony Wong }
	copyright {
Copyright (c) 1990, 1991, 1992 The Regents of the University of California.
All rights reserved.
See the file ~ptolemy/copyright for copyright notice,
limitation of liability, and disclaimer of warranty provisions.
	}
	location { CG96 dsp library }
        explanation {
.PP
This star estimates a certain number of samples
of the cross correlation of the 2 inputs
by averaging a certain number of input samples.
The number of outputs depend on the values of the parameters.
.IP
If \fInoLags\fR=0, then the star produces only 1 output,
which corresponds to the power of the process.
.IP
If \fInoLags\fR!=0 and \fIsymmetric\fR is set to "Yes",
then the star produces (2*\fInoLags\fR) outputs.
The output is symmetric about n=0
and can be used for power spectrum calculations.
.IP
If \fInoLags\fR!=0 and \fIsymmetric\fR is set to "No",
then the star produces (1+\fInoLags\fR) outputs.
The output can be used as input for the Levinson-Durbin star.
.PP
This block can also estimate the autocorrelation
of a sequence by connecting the 2 inputs to the same source.
.PP
If the parameter \fIunbiased\fR is set to "No",
then the correlation is given by:
.br
.nf
        r(k) = 1/(N) * Sum(n=k...N-1)[x(n) * y(n-k)]
.fi
for k=0,...,p;
where N is the number of input to average (\fInoInputsToAvg\fR)
and p is the number of lags to estimate (\fInoLags\fR).
This estimate is biased because the outermost lags
have fewer than N terms in the summation,
and yet the summation is still normalized by N.
.PP
If the parameter \fIunbiased\fR is set to "Yes" (the default),
then the correlation is given by:
.br
.nf
        r(k) = 1/(N-k) * Sum(n=k...N-1)[x(n) * y(n-k)]
.fi
for k=0,...,p;
where N is the number of input to average (\fInoInputsToAvg\fR)
and p is the number of lags to estimate (\fInoLags\fR).
In this case, the estimate is unbiased.
However, note that if the block is used for
autocorrelation estimation, the unbiased estimate does not
guarantee a positive definite sequence,
so a power spectral estimate based on this estimate
may have negative components.
	}
        seealso { CG96Average }

	input {
		name{xin}
		type{float}
	}
	input {
		name{yin}
		type{float}
	}
	output {
		name{output}
		type{float}
    	}
        state {
                name {noInputsToAvg}
                type {int}
                default {"64"}
                desc {Number of input samples to average.}
        }
        state {
                name {noLags}
                type {int}
                default {"16"}
                desc { Number of autocorrelation lags to output.}
        }
        state {
                name {unbiased}
                type {int}
                default {"YES"}
                desc { If YES, the estimate will be unbiased.}
        }
        state {
                name {symmetric}
                type {int}
                default {"YES"}
                desc { If YES, output is symmetric about n.}
        }
	state {
		name {copy}
		type {floatarray}
		default {"0"}
		desc {internal}
	        attributes {A_YMEM|A_NONCONSTANT|A_NONSETTABLE|A_NOINIT}
        }
	state {
		name {oldoutput}
		type {floatarray}
		default {"0"}
		desc {internal}
	        attributes {A_YMEM|A_NONCONSTANT|A_NONSETTABLE|A_NOINIT}
        }
	state {
		name {factors}
		type {floatarray}
		default {"0"}
		desc {internal}
	        attributes {A_YMEM|A_NONCONSTANT|A_NONSETTABLE|A_NOINIT}
        }
        state {
                name {X}
	        type {float}
	        default {0}
	        desc {internal}
	        attributes {A_NONCONSTANT|A_NONSETTABLE}
        }	       
	setup {
		if (int (noInputsToAvg) <= int (noLags)) {
	               Error::abortRun(*this,
				 ": noLags is larger than noInputsToAvg.");
			return;
		}
		xin.setSDFParams(int (noInputsToAvg),int (noInputsToAvg) - 1);
		yin.setSDFParams(int (noInputsToAvg),int (noInputsToAvg) - 1);
		if (noLags == 0)
			output.setSDFParams(1, 0);
		else if (symmetric)
			output.setSDFParams(2*int(noLags),2*int(noLags) - 1);
		else
			output.setSDFParams(int (noLags) + 1, int (noLags));

		if (noLags > 1)
			copy.resize(noInputsToAvg);
		else
			copy.resize(0);

		if (symmetric) {
			if (int (noLags) - 1 > 0)
				oldoutput.resize(int (noLags) - 1);
			else
				oldoutput.resize(0);
		} else
			oldoutput.resize(0);

		if (unbiased)
			factors.resize(int (noLags));
		else
			factors.resize(0);
	}

	initCode {
		addCode(org);
		if (unbiased)
			for (int j = 1; j <= noLags; j++) {
				X = 1.0 / (int (noInputsToAvg) - j);
				addCode(dc);
			}
		addCode(orgp);
	}

        codeblock(org) {
        org     $ref(factors)
        }
        codeblock(dc) {
        dc      $val(X)
        }
        codeblock(orgp) {
        org     p:
        }

        codeblock(zero) {
        move    #<$addr(xin),r0
        move    #>$addr(yin),r2
        fclr    d0      x:(r0)+,d4.s
        fclr    d2      x:(r2)+,d6.s
        do      #$val(noInputsToAvg),$label(inloop)
        fmpy.s  d4,d6,d2        x:(r0)+,d4.s
        fadd.s  d2,d0           x:(r2)+,d6.s
$label(inloop)
        move    #>$val(X),d1.s
        fmpy.s  d0,d1,d0
        move    d0.s,$ref(output)
        }    
        codeblock(notZero) {
; initialize pointers
        move    #<$addr(xin),r0
        move    #>$addr(yin),r2
        move    #>$addr(copy),r4
        move    #>$addr(output),r6
; copy yin to y memory and compute r(0)
        fclr    d0      x:(r0)+,d4.s
        fclr    d2      x:(r2)+,d6.s
        do      #$val(noInputsToAvg),$label(inloop)
        fmpy.s  d4,d6,d2        x:(r0)+,d4.s    d6.s,y:(r4)+
        fadd.s  d2,d0           x:(r2)+,d6.s
$label(inloop)
        move    #>$val(X),d1.s
        fmpy.s  d0,d1,d0
        move    d0.s,x:(r6)+
        }
        codeblock(sym) {
        move    #>$addr(oldoutput),r1
        }
        codeblock(unb) {
        move    #>$addr(factors),r3
        }
        codeblock(reg) {
        move    #>$val(X),d1.s
        }
        codeblock(cont) {
        move    #1,d5.l
        do      #$val(noLags)-1,$starSymbol(lab)
        move    #<$addr(xin),d3.l
        move    #>$addr(copy),r4
        add     d5.l,d3.l
        move    d3.l,r0
        move    #>$val(noInputsToAvg)-1,d7.l
        sub     d5.l,d7.l
        fclr    d0      x:(r0)+,d4.s
        fclr    d2      y:(r4)+,d6.s
        rep     d7.l
        fmpy    d4,d6,d2        fadd.s  d2,d0   x:(r0)+,d4.s    y:(r4)+,d6.s
        fmpy    d4,d6,d2        fadd.s  d2,d0
        fadd.s  d2,d0
        inc     d5.l
        }
        codeblock(unb1) {
        move    y:(r3)+,d1.s
        }
        codeblock(cont1) {
        fmpy.s  d0,d1,d0
        move    d0.s,x:(r6)+
$starSymbol(lab)
        }
        codeblock(sym1) {
        fmpy.s  d0,d1,d0
        move    d0.s,x:(r6)+
        move    d0.s,y:(r1)+
$starSymbol(lab)
        }
        codeblock(cont2) {
; compute r(no_lags)
        move    #>$addr(xin)+$val(noLags),r0
        move    #<$addr(copy),r4
        fclr    d0      x:(r0)+,d4.s
        fclr    d2      y:(r4)+,d6.s
        rep     #$val(noInputsToAvg)-$val(noLags)-1
        fmpy    d4,d6,d2        fadd.s  d2,d0   x:(r0)+,d4.s    y:(r4)+,d6.s
        fmpy    d4,d6,d2        fadd.s  d2,d0
        fadd.s  d2,d0
        }
        codeblock(unb2) {
        move    y:(r3),d1.s
        }
        codeblock(term) {        
        fmpy.s  d0,d1,d0
        move    d0.s,x:(r6)+
        }
        codeblock(outerloop) {
; output the other half
        do      #$val(noLags)-1,$label(outerloop)
        move    y:-(r1),d0.s
        move    d0.s,x:(r6)+
$label(outerloop)
        }    

	go {
		X = 1.0 / double (noInputsToAvg);
		if (noLags == 0)
			addCode(zero);
		else
			addCode(notZero);

		if (noLags > 1) {
			if (symmetric)
				addCode(sym);
			if (unbiased)
				addCode(unb);
			else
				addCode(reg);
			addCode(cont);
			if (unbiased)
				addCode(unb1);
			if (symmetric)
				addCode(sym1);
			else
				addCode(cont1);
			addCode(cont2);
			if (unbiased)
				addCode(unb2);
			else
				addCode(reg);
			addCode(term);
		}
		if ((symmetric) && (int (noLags) > 1))
			addCode(outerloop);
	}

	execTime {
		int             a = 0;
		int             b = 0;
		int             c = 0;

		if (int (noLags) == 0)
			a = 12 + (2 * int (noInputsToAvg));
		else {
			a = (14 + (3*int(noInputsToAvg)) +
			          (17 * int (noLags)));
			if (unbiased)
				b = noLags;
			if (symmetric)
				c = 1 + 3 * int (noLags);
		}

		return a + b + c;
	}
}

