/*
 * List Management Package Header File
 *
 * David Harrison
 * University of California, 1985-1988
 *
 * This file contains public type definitions for the List Managment
 * package implemented in list.c.  This is stand alone package.  See
 * list.doc for an interface description.
 */

#ifndef LS_DEFINED
#define LS_DEFINED

#include "ansi.h"
#include "copyright.h"

/* Package name for error handling */
extern char *ls_pkg_name;

/* This can be typedef'ed to void if supported */
typedef struct ls_dummy_defn {
    int dummy;			/* Not used */
} ls_dummy;

typedef ls_dummy *lsList;	/* List handle           */
typedef ls_dummy *lsGen;	/* List generator handle */
typedef ls_dummy *lsHandle;	/* Handle to an item     */
typedef int lsStatus;		/* Return codes          */
typedef char *lsGeneric;	/* Generic pointer       */

#define	LS_NIL		0	/* Nil for lsList       */

#define LS_BADSTATE	-3	/* Bad generator state   */
#define LS_BADPARAM	-2	/* Bad parameter value   */
#define LS_NOMORE	-1	/* No more items         */

#define	LS_OK		0

#define LS_BEFORE	1	/* Set spot before object */
#define LS_AFTER	2	/* Set spot after object  */
#define LS_STOP		3	/* Stop generating items  */
#define LS_DELETE	4	/* Delete generated item  */

/*
 * Fatal error conditions
 */

#define LS_NO_MEM	0	/* No more memory   */
#define LS_BAD_ARG	1	/* Bad argument     */
#define LS_NO_ITEM	2	/* Missing item     */
#define LS_CORRUPT	3	/* Corrupt list     */
#define LS_BADRET	4	/* Bad return value */
#define LS_UNKNOWN	99	/* Unknown error    */

/*
 * For all those routines that take a handle,  this macro can be
 * used when no handle is required.
 */

#define LS_NH		(lsHandle *) 0

extern lsList lsCreate();
  /* Create a new list */

extern void lsDestroy
  ARGS((lsList list, void (*delFunc)()));
  /* Delete a previously created list */

extern lsList lsCopy
  ARGS((lsList list, lsGeneric (*copyFunc)()));
  /* Copies the contents of a list    */

extern lsList lsAppend
  ARGS((lsList list1, lsList list2));
  /* Destructively appends two lists  */

extern lsStatus lsFirstItem
  ARGS((lsList list, lsGeneric *data, lsHandle *itemHandle));
  /* Gets the first item of a list */

extern lsStatus lsLastItem
  ARGS((lsList list, lsGeneric *data, lsHandle *itemHandle));
  /* Gets the last item of a list */

extern void lsNewBegin
  ARGS((lsList list, lsGeneric data, lsHandle *itemHandle));
  /* Add item to start of list */

extern void lsNewEnd
  ARGS((lsList list, lsGeneric data, lsHandle *itemHandle));
  /* Add item to end of list */

extern lsStatus lsDelBegin
  ARGS((lsList list, lsGeneric *data));
  /* Delete first item of a list */

extern lsStatus lsDelEnd
  ARGS((lsList list, lsGeneric *data));
  /* Delete last item of a list */

extern int lsLength
  ARGS((lsList list));
  /* Returns the length of the list */

extern lsGen lsStart
  ARGS((lsList list));
  /* Begin generation of items in a list */

extern lsGen lsEnd
  ARGS((lsList list));
  /* Begin generation at end of list */

extern lsGen lsGenHandle
  ARGS((lsHandle itemHandle, lsGeneric *data, int option));
  /* Produces a generator given a handle */

extern lsStatus lsNext
  ARGS((lsGen generator, lsGeneric *data, lsHandle *itemHandle));
  /* Generate next item in sequence */

extern lsStatus lsPrev
  ARGS((lsGen generator, lsGeneric *data, lsHandle *itemHandle));
  /* Generate previous item in sequence */

extern void lsInBefore
  ARGS((lsGen generator, lsGeneric data, lsHandle *itemHandle));
  /* Insert an item before the most recently generated by lsNext */

extern void lsInAfter
  ARGS((lsGen generator, lsGeneric data, lsHandle *itemHandle));
  /* Insert an item after the most recently generated by lsNext  */

extern lsStatus lsDelBefore
  ARGS((lsGen generator, lsGeneric *data));
  /* Delete the item before the current spot */

extern lsStatus lsDelAfter
  ARGS((lsGen generator, lsGeneric *data));
  /* Delete the item after the current spot */

extern void lsFinish
  ARGS((lsGen generator));
  /* End generation of items in a list */

/* As a convenience */

#define lsForeachItem(list, gen, data)				\
    for(gen = lsStart(list); 					\
	(lsNext(gen, (lsGeneric *) &data, LS_NH) == LS_OK) 	\
	    || ((void) lsFinish(gen), 0); )

/* Functional list generation */
extern lsStatus lsForeach
  ARGS((lsList list, lsStatus (*userFunc)(), lsGeneric arg));
   /* Generates items in a list forward calling a function for each item */

extern lsStatus lsBackeach
  ARGS((lsList list, lsStatus (*userFunc)(), lsGeneric arg));
   /* Generates items in a list backward calling a function for each item */

extern lsList lsQueryHandle
  ARGS((lsHandle itemHandle));
  /* Returns the list of a handle */

extern lsGeneric lsFetchHandle
  ARGS((lsHandle itemHandle));
  /* Returns data associated with handle */

extern void lsRemoveItem
  ARGS((lsHandle itemHandle, lsGeneric *userData));
  /* Removes item associated with handle from list */

extern void lsSort
  ARGS((lsList list, int (*compare)()));
  /* Sorts a list */

extern void lsUniq
  ARGS((lsList list, int (*compare)(), void (*delFunc)()));
  /* Removes duplicates from a sorted list */

#endif
