/** @file tree.h
 * Directory tree handling and file system utility functions
 * @author Marko Mkel <marko.makela@iki.fi>
 */

/* Copyright  2003,2006 Marko Mkel.

   This file is part of PHOTOMOLO, a program for generating
   thumbnail images and HTML files for browsing digital photographs.

   PHOTOMOLO is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   PHOTOMOLO is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   The GNU General Public License is often shipped with GNU software, and
   is generally kept in a file called COPYING or LICENSE.  If you do not
   have a copy of the license, write to the Free Software Foundation,
   59 Temple Place, Suite 330, Boston, MA 02111 USA. */

#include <sys/types.h>

#if defined WIN32 || defined __WIN32
# undef HAVE_LINK
#else
# define HAVE_LINK
#endif

/** types of files */
enum file_type {
  FILE_JPEG,	/**< JFIF or Exif JPEG image */
  FILE_IMAGE,	/**< other image file that can be converted */
  FILE_OTHER	/**< non-image file */
};

/** File information entry */
struct fileinfo
{
  /** file name */
  char* name;
  /** modification time */
  time_t mtime;
  /** type of file */
  enum file_type kind;
  /** converted file name (or NULL) */
  char* converted;
};

/** Directory tree.
 * The first two fields must be common with struct fileinfo.
 */
struct dirtree
{
  /** path name of the directory */
  char* name;
  /** modification time of last image contained in the subtree */
  time_t mtime;
  /** number of subdirectories */
  unsigned n_dirs;
  /** number of all files in the directory */
  unsigned n_files;
  /** number of JPEG image files */
  unsigned n_jpeg;
  /** number of non-JPEG image files */
  unsigned n_images;
  /** number of all files in subdirectories */
  unsigned n_files_dirs;
  /** number of JPEG image files in subdirectories */
  unsigned n_jpeg_dirs;
  /** number of non-JPEG image files in subdirectories */
  unsigned n_images_dirs;
  /** array of subdirectories */
  struct dirtree* dirs;
  /** array of files */
  struct fileinfo* files;
};

/** Initialize a directory tree structure
 * @param tree	the directory tree object
 * @param name	the path name of the directory
 */
void
inittree (struct dirtree* tree,
	  const char* name);

/** Read a directory tree
 * @param tree	the destination tree (with the path name filled in)
 * @param dir_p	callback for pruning directories by name
 * @param img_p	callback for identifying files by name
 * @return	zero on success, nonzero on failure
 */
int
readtree (struct dirtree* tree,
	  int (*dir_p) (const char*),
	  enum file_type (*img_p) (const char*));

/** Deallocate a directory tree
 * @param tree	the directory tree
 */
void
freetree (struct dirtree* tree);

/** Determine if the specified file exists
 * @param name	the path name of the file
 * @param mtime	the modification time
 * @return	nonzero if the file exists with the given date stamp
 */
int
exists_mtime (const char* name,
	      time_t mtime);

/** Touch a file to the specified modification time
 * @param name	the path name of the file
 * @param mtime	the modification time
 * @return	zero on success, nonzero on failure
 */
int
touch_mtime (const char* name,
	     time_t mtime);

/** Create a directory
 * @param name	the path name of the directory
 * @return	zero on success, nonzero on failure
 */
int
make_directory (const char* name);

/** Remove a directory tree
 * @param name	the path name of the directory
 * @return	zero on success, nonzero on failure
 */
int
remove_directory (const char* name);

#ifdef HAVE_LINK
/** Create a hard or soft (symbolic) link
 * @param dname	the path name of the link destination
 * @param name	the path name of the link source
 * @param sym	nonzero=create symbolic link; zero=hard link
 * @return	zero on success, nonzero on failure
 */
int
create_link (const char* dname, const char* name, int sym);
#endif /* HAVE_LINK */
