/** @file jpegcom.c
 * Read or write comments in JFIF or EXIF JPEG files
 * @author Marko Mkel <marko.makela@iki.fi>
 */

/* Copyright  2003,2004 Marko Mkel.

   This file is part of JPEGCOM, a program for reading and writing
   comments in digital photographs.

   JPEGCOM is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   JPEGCOM is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   The GNU General Public License is often shipped with GNU software, and
   is generally kept in a file called COPYING or LICENSE.  If you do not
   have a copy of the license, write to the Free Software Foundation,
   59 Temple Place, Suite 330, Boston, MA 02111 USA. */

#if defined WIN32 || defined __WIN32
# undef __STRICT_ANSI__
# include <sys/types.h>
# include <sys/utime.h>
#else
# ifndef _POSIX_SOURCE
#  define _POSIX_SOURCE
# endif
# include <sys/types.h>
# include <utime.h>
#endif
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <locale.h>

#include "rename.h"

/** Read 16 bits from file, most significant byte first
 * @param file	the file
 * @return	the data read
 */
static unsigned
read_word (FILE* file)
{
  unsigned word;
  word = (unsigned) getc (file);
  word <<= 8;
  word |= (unsigned) getc (file);
  return word;
}

/** Find the next JPEG marker code
 * @param file	the file
 * @return	the marker code found, or EOF
 */
static int
read_marker (FILE* file)
{
  int c;
  /* find the next 0xff */
  do {
    c = getc (file);
    if (c == EOF)
      return c;
  }
  while (c != 0xff);
  /* find the marker, ignoring subsequent 0xff bytes */
  do {
    c = getc (file);
    if (c == EOF)
      return c;
  }
  while (c == 0xff);
  return c;
}

/** Write a character in printable form
 * @param c	the character to be printed
 * @param f	the output file
 * @return	zero on success, nonzero on failure
 */
static int
write_char (int c, FILE* f)
{
  if (c == (int) '"' || c == (int) '\\')
    return (int) '\\' != putc ((int) '\\', f) || c != putc (c, f);
  if (c == (int) '\n' || c == (int) '\t' || isprint (c))
    return c != putc (c, f);
  return 4 != fprintf (f, "\\%03o", (unsigned) c);
}

/** Read JPEG image comments
 * @param in	the image file, without the SOI marker
 * @param com	the output file for the comments
 * @return	NULL on success, error message on failure
 */
static const char*
readjpegcom (FILE* in,
	     FILE* com)
{
  /* scan markers until start of scan (compressed data) */
  for (;;) {
    unsigned length;
    int marker = read_marker (in);
    if (marker == EOF) {
    eof:
      return "premature end of file";
    }
    else if (marker == 0xd9/* end of image */ ||
	     marker == 0xda/* start of scan (compressed data) */)
      return 0;
    length = read_word (in);
    if (length < 2)
      return "erroneous JPEG marker length";
    length -= 2;
    if (marker == 0xfe) {
      /* comment */
      if ((int) '"' != putc ((int) '"', com)) {
      output:
	return "write error";
      }
      while (length--) {
	int c = getc (in);
	if (c == EOF) {
	  (void) fputs ("\"\n", com);
	  goto eof;
	}
	else if (write_char (c, com))
	  goto output;
      }
      if (EOF == fputs ("\"\n", com))
	goto output;
    }
    else if ((marker & 0xf0) == 0xc0 &&
	     marker != 0xc4 && marker != 0xcc) {
      /* start of frame marker (image dimensions) */
      /** image dimensions */
      unsigned height, width;
      /** number of colour components */
      int comps;
      (void) getc (in); /* skip data precision */
      height = read_word (in);
      width = read_word (in);
      comps = getc (in);
      if (comps == EOF)
	goto eof;
      if (length != (6 + ((unsigned) comps) * 3))
	return "inconsistent SOF marker length";
      (void) fprintf (com, "# %ux%u\n", width, height);
      for (length -= 6; length--; )
	if (getc (in) == EOF)
	  goto eof;
    }
    else
      while (length--)
	if (getc (in) == EOF)
	  goto eof;
  }
}

/** Write 16 bits to file, most significant byte first
 * @param word	the data
 * @param file	the file
 * @return	zero on error
 */
static int
write_word (unsigned word,
	    FILE* file)
{
  return
    putc (word >> 8, file) == EOF ||
    putc (word & 0xff, file) == EOF;
}

/** Read a double quoted string
 * @param file	the input file
 * @param s	the input string (optional, for reallocated storage)
 * @param size	(output) length of the string (optional)
 * @return	the string, or NULL on failure
 */
static char*
read_name (FILE* file,
	   /*@null@*/char* s,
	   /*@null@*/ /*@out@*/unsigned* size)
{
  /** allocated length of the string */
  unsigned len;
  /** reallocated string */
  char* t;
  /** character read from the file */
  int c;
  /* skip whitespace and read the leading quote */
  while ((c = getc (file)) != (int) '"') {
    if (c == (int) '#') {
      /* skip until end of line or end of file */
      do
	c = getc (stdin);
      while (c != EOF && c != (int) '\n');
    }
    if (c == EOF) {
      (void) fputs ("unexpected end of file while looking for string\n",
		    stderr);
      return 0;
    }
    if (!isspace (c)) {
      (void) fputs ("unexpected character while looking for string\n",
		    stderr);
      return 0;
    }
  }
  if (!(t = realloc (s, len = 1))) {
  memory:
    (void) fputs ("out of memory for the string\n", stderr);
    free (s);
    return 0;
  }
  s = t;
  /* read the string */
  while ((c = getc (file)) != EOF) {
  again:
    if (c == (int) '"') {
      s[--len] = 0;
      if (size)
	*size = len;
      return s;
    }
    if (!(len & (len + 1))) {
      if (!(t = realloc (s, (len + 1) << 1)))
	goto memory;
      s = t;
    }
    if (c == (int) '\\') {
      if ((c = getc (file)) == EOF)
	break;
      else if (c >= (int) '0' && c <= (int) '7') {
	/* got octal digit, read up to 3 digits */
	int outch = c - (int) '0';
	if (EOF == (c = getc (file)))
	  break;
	if (c < (int) '0' || c > (int) '7')
	  goto done;
	outch <<= 3; outch |= c - (int) '0';
	if (EOF == (c = getc (file)))
	  break;
	if (c < (int) '0' || c > (int) '7')
	  goto done;
	outch <<= 3; outch |= c - (int) '0';
      done:
	s[len++ - 1] = (char) outch;
	if (c < (int) '0' || c > (int) '7')
	  goto again;
	continue;
      }
      /* fall through */
    }
    s[len++ - 1] = (char) c;
  }
  (void) fputs ("unexpected end of file while reading string\n", stderr);
  free (s);
  return 0;
}

/** Write image comments
 * @param in	the input image file
 * @param out	the output image file
 * @param com	the input file for the comments
 * @return	NULL on success, error message on failure
 */
static const char*
writecom (FILE* in,
	  FILE* out,
	  FILE* com)
{
  if (read_word (in) != 0xffd8)
    return "not a JPEG image file";
  if (write_word (0xffd8, out)) {
  output:
    return "write error";
  }
  /* scan markers until start of scan (compressed data) */
  for (;;) {
    unsigned length;
    int marker = read_marker (in);
    if (marker == EOF) {
    eof:
      return "premature end of JPEG file";
    }
    else if (marker == 0xd9/* end of image */ ||
	     marker == 0xda/* start of scan (compressed data) */) {
      /* write the comments and copy the rest of in to out */
      /** comment string */
      char* comment = 0;
      for (;;) { /* write the comments */
	int c = getc (com);
	if (c == (int) '#') {
	  /* skip until end of line or end of file */
	  do
	    c = getc (stdin);
	  while (c != EOF && c != (int) '\n');
	}
	if (c == EOF)
	  break;
	if (isspace (c))
	  continue;
	(void) ungetc (c, com);
	if (c != (int) '"' || !(comment = read_name (com, comment, &length)))
	  break;
	if (write_word (0xfffe, out) ||
	    write_word (length + 2, out) ||
	    length != fwrite (comment, 1, length, out))
	  goto output;
      }
      free (comment);

      /* copy the rest of the input image to output */
      if (write_word (0xff00 | marker, out))
	goto output;
      for (;;) {
	static char buf[8192];
	size_t len = fread (buf, 1, sizeof buf, in);
	if (len != fwrite (buf, 1, len, out))
	  goto output;
	if (len < sizeof buf)
	  break;
      }
      return 0;
    }

    /* process the marker */
    length = read_word (in);
    if (length < 2)
      return "erroneous JPEG marker length";
    if (marker == 0xfe) {
      /* ignore comments in the input file */
      length -= 2;
      while (length--)
	if (getc (in) == EOF)
	  goto eof;
      continue;
    }
    /* copy other than comment markers */
    if (write_word (0xff00 | marker, out) ||
	write_word (length, out))
      goto output;
    length -= 2;
    while (length--) {
      int c = getc (in);
      if (c == EOF)
	goto eof;
      if (c != putc (c, out))
	goto output;
    }
  }
}

/** Write a name, escaped in C style
 * @param name	the name (NUL terminated)
 * @param file	the output file
 * @return	zero on success
 */
static int
write_name (const char* name,
	    FILE* file)
{
  for (; *name; *name++)
    if (write_char ((unsigned char) *name, file))
      return EOF;
  return 0;
}

/** Copy the timestamp of a file to another
 * @param name	the file name whose timestamp is to be rewritten
 * @param ref	the reference file
 * @return	zero if successful
 */
static int
touch_file (const char* name,
	    const char* ref)
{
  int status;
  struct stat statbuf;
  if ((status = stat (ref, &statbuf))) {
    (void) fputs (ref, stderr), (void) fflush (stderr);
    perror (": stat");
  }
  else {
    struct utimbuf utim;
    utim.actime = utim.modtime = statbuf.st_mtime;
    if ((status = utime (name, &utim))) {
      (void) fputs (ref, stderr), (void) fflush (stderr);
      perror (": utime");
    }
  }
  return status;
}

/** Remove a hard link to a file
 * @param name	the file name to be removed
 * @return	zero if successful
 */
static int
remove_file (const char* name)
{
  int status = unlink (name);
  if (status) {
    (void) fputs (name, stderr), (void) fflush (stderr);
    perror (": unlink");
  }
  return status;
}

/** Main program
 * @param argc	argument count
 * @param argv	argument vector
 * @return	zero on success
 */
int
main (int argc, char** argv)
{
  setlocale (LC_ALL, "");

  if (argc > 1) {
    /* read the comments of the listed files */
    int i;
    if (argc == 2 &&
	fputs ("#This is a batch file for"
	       " commenting, renaming and copying image files.\n"
	       "#Comment lines start with '#'.\n"
	       "#Typical usage:"
	       " jpegcom *.jpg > file; $EDITOR file; jpegcom < file\n"
	       "#Files can be renamed or copied by editing this file.\n"
	       "#Comments are written after the second name.\n"
	       "#The characters '\"' and '\\'"
	       " must be written '\\\"' and '\\\\'.\n"
	       "#Example:\n"
	       "#\trename \"IMG_0001.JPG\"\t\"potato.jpg\"\n"
	       "#\t\"A \\\"beautiful\\\"\n"
	       "#\tflower\"\n"
	       "#\t\"Solanum tuberosum\"\n"
	       "#\tcopy \"IMG_0002.JPG\"\t\"onion.jpg\"\n"
	       "#\t\"Allium cepa\"\n", stdout) == EOF) {
    output:
      perror ("write(stdout)");
      return 1;
    }
    for (i = 1; i < argc; i++) {
      FILE* f = fopen (argv[i], "rb");
      const char* s;
      if (!f) {
	(void) fputs (argv[i], stderr), (void) fflush (stderr);
	perror (": open (reading)");
	return 1;
      }
      if (read_word (f) != 0xffd8) {
	(void) fputs (argv[i], stderr);
	(void) fputs (": not a JPEG image file\n", stderr);
      }
      else if (fputs ("copy \"", stdout) == EOF ||
	       write_name (argv[i], stdout) ||
	       fputs ("\"\t\"", stdout) == EOF ||
	       write_name (argv[i], stdout) ||
	       fputs ("\"\n", stdout) == EOF) {
	fclose (f);
	goto output;
      }
      else if ((s = readjpegcom (f, stdout))) {
	(void) fputs (argv[i], stderr);
	(void) fputs (": ", stderr);
	(void) fputs (s, stderr);
	(void) putc ((int) '\n', stderr);
      }
      (void) fclose (f);
    }
  }
  else {
    /* process the listed files */
    int c;
    while ((c = getc (stdin)) != EOF) {
      if (c == (int) '#') {
	/* skip until end of line */
	for (;;) {
	  if ((c = getc (stdin)) == EOF)
	    return 0;
	  if (c == (int) '\n')
	    break;
	}
      }
      else if (!isspace (c)) {
	/* look for "rename" or "copy" */
	static const char *rename = "ename";
	static const char *copy = "opy";
	const char* r;
	int do_remove = c == (int) 'r';
	char* name1; char* name2 = 0;
	if (c == (int) 'r')
	  r = rename;
	else if (c == (int) 'c')
	  r = copy;
	else {
	  (void) fputs ("syntax error: expected \"rename\" or \"copy\", got ",
		 stderr);
	  goto syntax;
	}
	do {
	  if ((c = getc (stdin)) != (int) *r) {
	    (void) fputs ("syntax error: expected \"", stderr);
	    (void) fputs (r, stderr);
	    (void) fputs ("\", got ", stderr);
	  syntax:
	    if (c == EOF)
	      (void) fputs ("EOF", stderr);
	    else
	      (void) putc (c, stderr);
	    (void) putc ((int) '\n', stderr);
	    return 2;
	  }
	}
	while (*++r);
	if ((name1 = read_name (stdin, 0, 0)) &&
	    (name2 = read_name (stdin, 0, 0))) {
	  int same = !strcmp (name1, name2);
	  FILE* in;
	  FILE* out;
	  int fd;
	  if (!(in = fopen (name1, "rb"))) {
	    (void) fputs (name1, stderr), (void) fflush (stderr);
	    perror (": open (reading)");
	    free (name1), free (name2);
	    return 1;
	  }
	  if (same) {
	    /* same name => change the last character of the name to '0' */
	    char* s;
	    for (s = name2; *s; s++);
	    s--;
	    *s = (int) '0';
	  }
	  (void) unlink (name2);
	  fd = open (name2,
#if defined WIN32 || defined __WIN32
		     O_WRONLY | O_CREAT | O_EXCL | O_BINARY, 0666
#else
		     O_WRONLY | O_CREAT | O_EXCL, 0444
#endif
		     );
	  if (fd < 0 || !(out = fdopen (fd, "wb"))) {
	    (void) fputs (name2, stderr), (void) fflush (stderr);
	    perror (": open (writing)");
	  cleanup:
	    free (name1), free (name2);
	    return 1;
	  }
	  if ((r = writecom (in, out, stdin))) {
	    (void) fputs (name1, stderr);
	    (void) fputs (": ", stderr);
	    (void) fputs (r, stderr);
	    (void) putc ((int) '\n', stderr);
	    (void) fclose (in), (void) fclose (out);
	    goto cleanup;
	  }
	  if (fclose (in)) {
	    (void) fputs (name1, stderr), (void) fflush (stderr);
	    perror (": close");
	    /* ignore failure to close input file */
	  }
	  if (fclose (out)) {
	    (void) fputs (name2, stderr), (void) fflush (stderr);
	    perror (": close");
	    /* abort on failure to close output file */
	    goto cleanup;
	  }
	  /* copy the timestamp and rename or remove the file */
	  if (touch_file (name2, name1) ||
	      (same
	       ? rename_file (name2, name1)
	       : (do_remove && remove_file (name1))))
	    goto cleanup;
	  free (name1), free (name2);
	}
	else {
	  free (name1), free (name2);
	  return 2;
	}
      }
    }
  }

  return 0;
}
