/** @file html.c
 * HTML output
 * @author Marko Mkel <marko.makela@iki.fi>
 */

/* Copyright  2003,2004,2005,2006 Marko Mkel.

   This file is part of PHOTOMOLO, a program for generating
   thumbnail images and HTML files for browsing digital photographs.

   PHOTOMOLO is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   PHOTOMOLO is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   The GNU General Public License is often shipped with GNU software, and
   is generally kept in a file called COPYING or LICENSE.  If you do not
   have a copy of the license, write to the Free Software Foundation,
   59 Temple Place, Suite 330, Boston, MA 02111 USA. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "tree.h"
#include "info.h"
#include "spec.h"
#include "html.h"

/** Write raw string to HTML
 * @param file	the output stream
 * @param s	the string
 * @return	zero on success, nonzero on error
 */
static int
html_raw (FILE* file,
	  const char* s)
{
  return fputs (s, file) == EOF;
}

/** Write appropriately escaped URI string to HTML
 * @param file	the output stream
 * @param s	the string
 * @return	zero on success, nonzero on error
 */
static int
html_uri (FILE* file,
	  const char* s)
{
  for (; *s; s++) {
    switch (*s) {
    default:
      if (*s > (char) 0x20 && *s < (char) 0x7e) {
	if (putc (*s, file) == EOF)
	  return 1;
	break;
      }
      /*@fallthrough@*/
    case '"': case '#': case '%': case '?': case ':': case '+':
    case '&': case '<': case '>':
      if (3 != fprintf (file, "%%%02x", (unsigned) (unsigned char) *s))
	return 1;
    }
  }
  return 0;
}

/** Write appropriately escaped string to HTML
 * @param file	the output stream
 * @param s	the string
 * @param e	pointer to end of string, or NULL
 * @return	zero on success, nonzero on failure
 */
static int
html_esc (FILE* file,
	  const char* s,
	  const char* e)
{
  for (; *s && (!e || s < e); s++) {
    const char* esc;
    switch (*s) {
    case '&': esc = "&amp;"; break;
    case '<': esc = "&lt;"; break;
    case '>': esc = "&gt;"; break;
    case '"': esc = "&quot;"; break;
    default:
      if (putc (*s, file) == EOF)
	return 1;
      continue;
    }
    if (fputs (esc, file) == EOF)
      return 1;
  }
  return 0;
}

/** Write appropriately escaped string to HTML
 * @param file	the output stream
 * @param s	the string, possibly containing shell meta-characters
 * @param e	pointer to end of string, or NULL
 * @return	zero on success, nonzero on failure
 */
static int
html_esc_meta (FILE* file,
	       const char* s,
	       const char* e)
{
  for (; *s && (!e || s < e); s++) {
    const char* esc;
    switch (*s) {
      /* HTML special characters */
    case '&': esc = "&amp;"; break;
    case '<': esc = "&lt;"; break;
    case '>': esc = "&gt;"; break;
    case '"': esc = "&quot;"; break;
      /* shell meta-characters */
    case '\t': case '\n': case '\r': case ' ':
    case '$': case '\'': case '(': case ')': case ';':
    case '\\': case '`': case '|':
      switch (fprintf (file, "&#%u;", (unsigned) *s)) {
      case 4: if ((unsigned) *s < 10) continue; break;
      case 5: if ((unsigned) *s >= 10 && (unsigned) *s < 100) continue; break;
      case 6: if ((unsigned) *s >= 100) continue; break;
      }
      return 1;
    default:
      if (putc (*s, file) == EOF)
	return 1;
      continue;
    }
    if (fputs (esc, file) == EOF)
      return 1;
  }
  return 0;
}

/** Output command-line arguments to HTML
 * @param file	the output stream
 * @param argc	number of command-line arguments
 * @param argv	command-line arguments
 * @return	zero on success, nonzero on failure
 */
static int
html_esc_args (FILE* file,
	       int argc,
	       char** argv)
{
  while (argc--)
    if (putc (' ', file) == EOF ||
	html_esc_meta (file, *argv++, 0))
      return 1;
  return 0;
}

/** Output image information in HTML
 * @param file	the output stream
 * @param mtime	file modification time
 * @param dim	image dimensions in a printable format
 * @param comm	image comments
 * @return	zero on success, nonzero on failure
 */
static int
html_info (FILE* file,
	   const char* mtime,
	   const char* dim,
	   const struct comment* comm)
{
  if (html_raw (file, dim) ||
      (*mtime && (html_raw (file, " ") ||
		  html_esc (file, mtime, 0))))
    return 1;
  for (; comm; comm = comm->next)
    if (html_raw (file, " \n") ||
	html_esc (file, comm->text, 0))
      return 1;
  return 0;
}

/** Generate links to alternative-resolution HTML files
 * @param file	the output stream
 * @param ixnam	index.html file name
 * @param thumb	available thumbnail resolutions
 * @param spec	thumbnail specification
 * @return	zero on success, nonzero on failure
 */
static int
html_link_specs (FILE* file,
		 const char* ixnam,
		 const struct thumb_spec* thumb,
		 const struct thumb_spec* spec)
{
  const struct thumb_spec* specs;
  for (specs = thumb; specs; specs = specs->next) {
    if (spec == specs)
      continue;
    if (html_raw (file, "<link title=\"") ||
	html_raw (file, specstr (specs)) ||
	html_raw (file, "\" rel=\"alternate\" href=\"../") ||
	html_uri (file, specs->thumbdir) ||
	html_raw (file, "/") ||
	html_uri (file, ixnam) ||
	html_raw (file, "\">\n"))
      return 1;
  }
  return 0;
}

/** Generate anchors to alternative-resolution HTML files
 * @param file	the output stream
 * @param ixnam	index.html file name
 * @param thumb	available thumbnail resolutions
 * @param spec	thumbnail specification
 * @return	zero on success, nonzero on failure
 */
static int
html_a_specs (FILE* file,
	      const char* ixnam,
	      const struct thumb_spec* thumb,
	      const struct thumb_spec* spec)
{
  const struct thumb_spec* specs;
  for (specs = thumb; specs; specs = specs->next) {
    if (spec == specs
	? (html_raw (file, specstr (specs)) ||
	   html_raw (file, "\n"))
	: (html_raw (file, "<a href=\"../") ||
	   html_uri (file, specs->thumbdir) ||
	   html_raw (file, "/") ||
	   html_uri (file, ixnam) ||
	   html_raw (file, "\">") ||
	   html_raw (file, specstr (specs)) ||
	   html_raw (file, "</a>\n")))
      return 1;
  }
  return 0;
}

/** Generate a link to a related directory listing
 * @param file	the output stream
 * @param rel	the relation (0="top")
 * @param spec	the thumbnail directory name
 * @param dir	the directory name
 * @param ixnam	index.html file name
 * @return	zero on success, nonzero on failure
 */
static int
html_link (FILE* file,
	   const char* rel,
	   const char* spec,
	   const char* dir,
	   const char* ixnam)
{
  {
    char* s = strrchr (dir, '/');
    if (s)
      dir = s + 1;
  }
  return
    html_raw (file, "<link title=\"") ||
    html_esc (file, dir, 0) ||
    html_raw (file, "\" rel=\"") ||
    html_raw (file, rel ? rel : "top") ||
    html_raw (file, "\" href=\"../../") ||
    (rel && (html_uri (file, dir) ||
	     html_raw (file, "/"))) ||
    html_uri (file, spec) ||
    html_raw (file, "/") ||
    html_uri (file, ixnam) ||
    html_raw (file, "\">\n");
}

/** Generate link to file in the root directory
 * @param file	the output stream
 * @param dir	the subdirectory relative to the root
 * @param css	index.css file name
 */
static int
html_uriref (FILE* file,
	     const char* dir,
	     const char* css)
{
  if (*css == '/' || strstr (css, "://")) /* absolute link */
    return html_raw (file, css);
  else {
    unsigned num_parents;
    const char* s;
    for (num_parents = 1, s = dir; *s; s++)
      if (*s == '/')
	num_parents++;
    while (num_parents--)
      if (html_raw (file, "../"))
	return 1;
    return html_uri (file, css);
  }
}

/** Generate links to parent directories
 * @param file	the output stream
 * @param dir	the directory name
 * @param ixnam	index.html file name
 * @param spec	the thumbnail directory name
 * @return	zero on success, nonzero on failure
 */
static int
html_parent_links (FILE* file,
		   const char* dir,
		   const char* ixnam,
		   const char* spec)
{
  unsigned num_parents;
  const char* s;
  char* e;
  for (num_parents = 0, s = dir; *s; s++)
    if (*s == '/')
      num_parents++;
  for (s = dir; num_parents--; s = e + 1) {
    unsigned i;
    e = strchr (s, '/');
    if (html_raw (file, "<a href=\""))
      return 1;
    for (i = num_parents + 2; i--; )
      if (html_raw (file, "../"))
	return 1;
    if (html_uri (file, spec) ||
	html_raw (file, "/") ||
	html_uri (file, ixnam) ||
	html_raw (file, "\">") ||
	html_esc (file, s, e) ||
	html_raw (file, "</a>/"))
      return 1;
  }
  return 0;
}

/** Generate links to child directories
 * @param file	the output stream
 * @param tree	the directory tree
 * @param pflen	number of first bytes to ignore in tree->name
 * @param ixnam	index.html file name
 * @param spec	the thumbnail directory name
 * @param blen	base directory name length
 * @return	zero on success, nonzero on failure
 */
static int
html_child_links (FILE* file,
		  const struct dirtree* tree,
		  unsigned pflen,
		  const char* ixnam,
		  const char* spec,
		  unsigned blen)
{
  unsigned i;
  if (!tree->n_dirs)
    return 0;
  if (html_raw (file, "<dl>\n"))
    return 1;
  for (i = 0; i < tree->n_dirs; i++) {
    const struct dirtree* dir = tree->dirs + i;
    const char* base = strrchr (dir->name + pflen, '/');
    if (base)
      base++;
    else
      base = dir->name + pflen;
    if (html_raw (file, "<dt><a href=\"../") ||
	html_uri (file, dir->name + pflen + blen + 1) ||
	html_raw (file, "/") ||
	html_uri (file, spec) ||
	html_raw (file, "/") ||
	html_uri (file, ixnam) ||
	html_raw (file, "\">") ||
	html_esc (file, base, 0) ||
	html_raw (file, "</a>"))
      return 1;
    if (dir->n_files || dir->n_files_dirs) {
      unsigned n_others =
	dir->n_files - dir->n_jpeg - dir->n_images;
      unsigned n_others_dirs =
	dir->n_files_dirs - dir->n_jpeg_dirs - dir->n_images_dirs;
      /* display the numbers of files in current directory */
      if ((0 > fprintf (file, " (%u", dir->n_jpeg)) ||
	  ((dir->n_images || n_others) &&
	   0 > fprintf (file, "+%u", dir->n_images)) ||
	  (n_others &&
	   0 > fprintf (file, "+%u", n_others)))
	return 1;
      /* display the numbers of files in subdirectories */
      if (dir->n_files_dirs) {
	if ((0 > fprintf (file, "; <em>%u", dir->n_jpeg_dirs)) ||
	    ((dir->n_images_dirs || n_others_dirs) &&
	     0 > fprintf (file, "+%u", dir->n_images_dirs)) ||
	    (n_others_dirs &&
	     0 > fprintf (file, "+%u", n_others_dirs)) ||
	    fputs ("</em>", file) == EOF)
	  return 1;
      }
      if (fputs (")", file) == EOF)
	return 1;
    }
    if (html_raw (file, "</dt>\n"))
      return 1;
    if (dir->n_dirs) {
      if (html_raw (file, "<dd>") ||
	  html_child_links (file, dir, pflen, ixnam, spec, blen) ||
	  html_raw (file, "</dd>\n"))
	return 1;
    }
  }
  return html_raw (file, "</dl>");
}

/** Dump the HTML code
 * @param file	the output stream
 * @param argc	number of command-line arguments
 * @param argv	command-line arguments
 * @param tree	the directory
 * @param pflen	number of first bytes to ignore in tree->name
 * @param level	number of '/' characters in the root directory name
 * @param info	image dimensions
 * @param comm	image comments
 * @param opts	dump options
 * @param thumb	available thumbnail resolutions
 * @param spec	the thumbnail resolution
 * @param top	name of parent directory
 * @param prev	name of previous directory
 * @param next	name of next directory
 * @return	zero on success, nonzero on failure
 */
int
dumphtml (FILE* file,
	  int argc,
	  char** argv,
	  const struct dirtree* tree,
	  unsigned pflen,
	  unsigned level,
	  const struct imageinfo* info,
	  const struct comment*const* comm,
	  const struct html_opts* opts,
	  const struct thumb_spec* thumb,
	  const struct thumb_spec* spec,
	  const char* top,
	  const char* prev,
	  const char* next)
{
  unsigned i;
  /** file name */
  char* fn;
  /** root directory name */
  const char* rootname;
  /** file modification time */
  char mtime[20];
  /** length of file name */
  unsigned fnlen = strlen (tree->name + pflen) + 1;
  /** length of the thumbnail directory name */
  unsigned speclen = strlen (spec->thumbdir) + 1;
  const char* title = strrchr (tree->name + pflen, '/');
  /** flag: listed any files yet? */
  int listed;

  if (title)
    title++;
  else
    title = tree->name + pflen;
  for (rootname = tree->name + pflen; level && *rootname; rootname++)
    if (*rootname == '/')
      level--;
  if (html_raw (file,
		"<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01//EN\">\n"
		"<html>\n"
		"<head>\n"
		"<meta http-equiv=\"Content-Type\""
		" content=\"text/html; charset=") ||
      html_esc (file, opts->charset, 0) ||
      html_raw (file, "\">\n"
		"<title>") ||
      html_esc (file, rootname, 0) ||
      html_raw (file, " (") ||
      html_raw (file, specstr (spec)) ||
      html_raw (file, ")</title>\n"
		"<meta name=\"Generator\" content=\"") ||
      html_raw (file, opts->generator) ||
      html_esc_args (file, argc, argv) ||
      html_raw (file, "\">\n"
		"<link rel=\"start\" href=\"") ||
      html_uriref (file, rootname, opts->html) ||
      html_raw (file, "\">\n"))
    return 1;
  if (opts->css &&
      (html_raw (file, "<link rel=\"stylesheet\" href=\"") ||
       html_uriref (file, rootname, opts->css) ||
       html_raw (file, "\">\n")))
    return 1;
  if (opts->icon &&
      (html_raw (file, "<link rel=\"icon\" href=\"") ||
       html_uriref (file, rootname, opts->icon) ||
       html_raw (file, "\">\n")))
    return 1;
  if (opts->msico &&
      (html_raw (file, "<link rel=\"SHORTCUT ICON\" href=\"") ||
       html_uriref (file, rootname, opts->msico) ||
       html_raw (file, "\">\n")))
    return 1;
  if ((top && html_link (file, 0, spec->thumbdir, top, opts->html)) ||
      (prev && html_link (file, "prev", spec->thumbdir, prev, opts->html)) ||
      (next && html_link (file, "next", spec->thumbdir, next, opts->html)) ||
      html_link_specs (file, opts->html, thumb, spec) ||
      html_raw (file,
		"</head>\n\n"
		"<body>\n<h1>") ||
      html_parent_links (file, rootname, opts->html, spec->thumbdir) ||
      html_esc (file, title, 0) ||
      html_raw (file, "</h1>\n") ||
      html_child_links (file, tree, pflen, opts->html, spec->thumbdir,
			strlen (tree->name + pflen)) ||
      html_raw (file, "<p>") ||
      html_a_specs (file, opts->html, thumb, spec) ||
      html_raw (file, "</p>\n"))
    return 1;

  if (!(fn = malloc (fnlen + speclen + 1))) {
  memory:
    free (fn);
    (void) fputs ("out of memory\n", stderr);
    return 1;
  }
  /* prepare the thumbnail file name */
  memcpy (fn, tree->name + pflen, fnlen - 1);
  memcpy (fn + fnlen, spec->thumbdir, speclen - 1);
  fn[fnlen - 1] = fn[fnlen + speclen - 1] = '/';

  /* list the image files */
  listed = 0;

  for (i = 0; i < tree->n_files; i++) {
    /** image dimensions */
    char dim[20];
    /** thumbnail image information */
    struct imageinfo thumb_info;
    /** thumbnail file name (converted or original) */
    const char* name;
    /** thumbnail file */
    FILE* f;
    /** thumbnail file name length */
    unsigned namelen;
    char* s;
    const struct fileinfo* const fi = tree->files + i;

    switch (fi->kind) {
    case FILE_JPEG:
    case FILE_IMAGE:
      if (!info[i].height || !info[i].width)
	continue; /* skip invalid file */
      break;
    case FILE_OTHER:
      continue;
    }

    if (!listed) {
      listed = 1;
      if (html_raw (file, "<p class=\"img\">"))
	goto error;
    }

    if (opts->notim ||
	!strftime (mtime, sizeof mtime, "%Y-%m-%d %H:%M:%S",
		   localtime (&fi->mtime)))
      *mtime = 0;

    name = fi->converted ? fi->converted : fi->name;
    sprintf (dim, "%u&times;%u", info[i].width, info[i].height);
    if (html_raw (file, "<a\nhref=\"../") ||
	html_uri (file, fi->name) ||
	html_raw (file, "\"><img src=\"") ||
	html_uri (file, name) ||
	html_raw (file, "\"\nalt=\"[") ||
	html_info (file, mtime, dim, comm[i]) ||
	html_raw (file, "]\"\ntitle=\"") ||
	html_info (file, mtime, dim, comm[i]) ||
	html_raw (file, "\""))
      goto error;
    namelen = strlen (name) + 1;
    s = realloc (fn, fnlen + speclen + namelen);
    if (!s)
      goto memory;
    fn = s;
    memcpy (fn + fnlen + speclen, name, namelen);
    if (!(f = fopen (fn, "rb"))) {
      (void) fputs (fn, stderr), (void) fflush (stderr);
      perror (": open (reading)");
      goto error;
    }
    s = (char*) readinfo (&thumb_info, 0, f);
    (void) fclose (f);
    if (s) {
      (void) fputs (fn, stderr);
      (void) fputs (": ", stderr), (void) fputs (s, stderr);
      (void) putc ('\n', stderr);
    }
    else
      (void) fprintf (file, "\nwidth=\"%u\" height=\"%u\"",
		      thumb_info.width, thumb_info.height);
    if (html_raw (file, "></a>"))
      goto error;
  }

  if (listed && html_raw (file, "</p>\n")) {
  error:
    free (fn);
    return 1;
  }

  /* list the non-image files */

  for (i = 0; i < tree->n_files; i++) {
    const struct fileinfo* const fi = tree->files + i;

    switch (fi->kind) {
    case FILE_JPEG:
    case FILE_IMAGE:
      continue; /* skip image files */
    case FILE_OTHER:
      break;
    }

    if (opts->notim ||
	!strftime (mtime, sizeof mtime, "%Y-%m-%d %H:%M:%S",
		   localtime (&fi->mtime)))
      *mtime = 0;

    if (html_raw (file, "<p><a href=\"../") ||
	html_uri (file, fi->name) ||
	(*mtime && (html_raw (file, "\"\ntitle=\"") ||
		    html_raw (file, mtime))) ||
	html_raw (file, "\">") ||
	html_esc (file, fi->name, 0) ||
	html_raw (file, "</a></p>\n"))
      goto error;
  }

  free (fn);
  return html_raw (file, "</body></html>\n");
}
