/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  double MSslMidMC (double Dxl, double Dxr, double Dyl, double Dyr)

Purpose:
  Calculate a slope for at a middle point (for fitting a cubic)

Description:
  This routine finds a slope that can be used to generate piecewise monotone
  cubic interpolants.  Consider three points (xl,yl), (x0,y0) and (xr,yr),
  where xl < x0 < xr.  The slope is calculated at the point (x0,y0).

Parameters:
  <-  double MSslMidMC
      Slope value at the reference point
   -> double Dxl
      Increment in abscissa value to the left of the reference point (x0-xl).
      This value must be positive.
   -> double  Dxr
      Increment in abscissa value to the right of the reference point (xr-x0).
      This value must be positive.
   -> double Dyl
      Increment in ordinate value to the left of the reference point (y0-yl).
   -> double Dyr
      Increment in ordinate value to the right of the reference point (yr-y0).

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.5 $  $Date: 1994/04/11 15:09:46 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: MSslMidMC.c 1.5 1994/04/11 libtsp-V2R7a $";

#include <libtsp.h>
#include <libtsp/nucleus.h>

#define ABSV(x)		(((x) < 0) ? -(x) : (x))

double
MSslMidMC (Dxl, Dxr, Dyl, Dyr)

     double Dxl;
     double Dxr;
     double Dyl;
     double Dyr;

{
  double Ds;
  double Smin, Smax, Sl, Sr;
  double a0, a1;

/* Check for increasing X values */
  if (Dxl <= 0.0 || Dxr <= 0.0)
    UThalt ("MSslMidMC: Abscissa values not in increasing order");

/* Sl is the slope to the left of point (x0,y0)
   Sr is the slope to the right of point (x0,y0) */
  Sl = Dyl / Dxl;
  Sr = Dyr / Dxr;

/* Use Brodlie modification of Butland formula (for same sign slopes)

              Sl Sr                  1       xl-x0
     d = --------------- , where a = - (1 + -------) .
         a Sl + (1-a) Sr             3      xl - xr

   This formula gives the harmonic mean of the slopes when x0-xl=xr-x0.  The
   factor a modifies the formula to take into account the relative spacing of
   the values (a varies from 1/3 to 2/3).  This formula satisfies the
   following properties.
   (a) the formula is invariant to an exchange of the the left and right
        points,
   (b) min(Sl,Sr) <= d <= max(Sl,Sr),
   (c) |d| <= 3 min(|sl|,|sr|).  This is a sufficient condition for
       monotonicity if d is calculated in this way for neighboring points.

  The following calculation is designed to avoid overflows, but is equivalent
  to the above formula.
         Sl Sr                  Smin
   ---------------  =  -------------------------
   a Sl + (1-a) Sr     a Sl/Smax + (1-a) Sr/Smax
  */

  if ((Sl > 0.0 && Sr > 0.0) || (Sl < 0.0 && Sr < 0.0)) {

    /* Slopes of the same sign */
    if (ABSV (Sl) > ABSV (Sr)) {
      Smax = Sl;
      Smin = Sr;
    }
    else {
      Smax = Sr;
      Smin = Sl;
    }
    Ds = Dxl + Dxr;
    a0 = (Ds + Dxr) / (3.0 * Ds);
    a1 = (Ds + Dxl) / (3.0 * Ds);
    return (Smin / ( a0 * (Sl / Smax) + a1 * (Sr / Smax) ));
  }

/*
   Slopes of opposite signs - force the derivative to zero to give a local
   maximum at the middle data point
*/
  else
    return 0.0;
}
