/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  void MSintMC (const float x[], const float y[], int N, const float xi[],
		float yi[], int Ni)

Purpose:
  Interpolate a table of values using a piecewise monotonic cubic interpolant

Description:
  This routine calculates interpolated values for a function defined by a table
  of reference data values.  The interpolated values are found by passing a
  piecewise monotonic cubic interpolant through the reference data values.  The
  monotonicity property guarantees that the interpolated data values do not go
  outside the range of the bracketing reference data values.

  The procedure defines the cubic interpolant between any two adjacent
  reference points in terms of the values and the derivatives at the reference
  points.  The derivatives at the reference points are determined by averaging
  the inverse slopes of the straight line segments joining the reference
  points to the left and right of a given reference point.  These slope values
  are changed if necessary to guarantee that the cubic interpolants are
  monotonic between reference points.  For instance a reference point is a
  local extremum (the slopes of the straight line segments joining reference
  points on either side of it are of different sign), the slope at that
  reference point is set to zero.  The slope of end points is determined from
  the slopes of the two straight lines joining the reference points beside the
  end point.

  References:
      Subroutine PCHIM, F. N. Fritsch, Lawrence Livermore National Laboratory.
      F. N. Fritsch and J. Butland, "A method for constructing local monotone
      piecewise cubic interpolants", SIAM J. Sci. Stat. Comput., vol. 5,
      pp. 300-304, June 1984.

Parameters:
   -> const float x[]
      Abscissa values for the reference points.  These values must be in
      increasing order.
   -> const float y[]
      Ordinate values for the reference points
   -> int N
      Number of reference points
   -> const float xi[]
      Abscissa values for the interpolated points.  These values must be
      bounded by x[0] and x[N-1].  The values xi[i] need not be in any
      particular order, although some efficiency is gained if they are in
      increasing or decreasing order.
  <-  float yi[]
      Resulting interpolated ordinate values
   -> int Ni
      Number of interpolated values

Author / revision:
  P. Kabal  Copyright (C) 1996
  $Revision: 1.12 $  $Date: 1996/05/06 18:26:04 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: MSintMC.c 1.12 1996/05/06 libtsp-V2R7a $";

#include <libtsp.h>
#include <libtsp/nucleus.h>

void
MSintMC (x, y, N, xi, yi, Ni)

     const float x[];
     const float y[];
     int N;
     const float xi[];
     float yi[];
     int Ni;

{
  int i, k, kp;
  double dh, dl;

/* Check the reference abscissa values */
  for (k = 1; k < N; ++k) {
    if (x[k] < x[k-1])
      UThalt ("MSintMC: Abscissa values not in increasing order");
  }

  /* Special case: only one reference point */
  if (N == 1) {
    for (i = 0; i < Ni; ++i) {
      if (xi[i] != x[0])
	UThalt ("MSintMC: Abscissa value %g not in range [%g, %g]",
		xi[i], x[0], x[0]);
      yi[i] = y[0];
    }
  }

  /* General case: 2 or more reference points */
  /* N reference points -> N-1 intervals */
  else {
    kp = -1;
    for (i = 0; i < Ni; ++i) {
      /* Search for the bracketing interval */
      k = SPquant (xi[i], x, N + 1);		/* N+1 quantizer regions */
      if (k == 0 && xi[i] == x[0])
	k = 1;
      if (k <= 0 || k >= N)			/* reject 2 outer regions */
	UThalt ("MSintMC: Abscissa value %g not in range [%g, %g]",
		xi[i], x[0], x[N-1]);

/* Interval found: x[k-1] <= xi[i] <= x[k], with 1 <= k <= N-1 */
/*
   Choose the derivatives at the bracketing points and then evaluate the cubic
   at the interpolation point
*/
      if (kp == k)
	;
      else if (kp == k - 1) {
	dl = dh;	/* dh has been defined previously */
	dh = MSslopeMC (k, x, y, N);
      }
      else if (kp == k + 1) {
	dl = MSslopeMC (k - 1, x, y, N);
	dh = dl;
      }
      else {
	dl = MSslopeMC (k - 1, x, y, N);
	dh = MSslopeMC (k, x, y, N);
      }
      yi[i] = MSevalMC (xi[i], x[k-1], x[k], y[k-1], y[k], dl, dh);
      kp = k;
    }
  }
  return;
}
