/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  int AFwriteHead (FILE *fp, long int offs, void *buf, int size, int Nelem,
                   int Swapb)

Purpose:
  Write and optionally swap audio file header values

Description:
  This routine writes data to an audio file header.  The information to be
  written is considered to be organized as Nelem elements each of size bytes.
  First, optionally each of the Nelem elements is byte swapped.  The data
  (Nelem * size bytes) is then written to the file.

Parameters:
  <-  int AFwriteHead
      Number of bytes written (equal to Nelem * size)
   -> long int offs
      Offset in bytes into the file
   -> const void *buf
      Pointer to a buffer of size Nelem * size
   -> int size
      Size of each element in bytes
   -> int Nelem
      Number of elements to be written
   -> int Swapb
      Byte swap flag.  This parameter is not used for size = 1.  If the bytes
      are to be swapped, size must be 2, 4 or 8 bytes.
      DS_EB     - File data is in big-endian byte order.  The data will be
      	          swapped if the current host uses little-endian byte order.
      DS_EL     - File data is in little-endian byte order data.  The data will
                  be swapped if the current host uses big-endian
		  byte order.
      DS_NATIVE - File data is in native byte order
      DS_SWAP   - File data is byte-swapped

Author / revision:
  P. Kabal  Copyright (C) 1995
  $Revision: 1.6 $  $Date: 1995/09/26 13:37:35 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: AFwriteHead.c 1.6 1995/09/26 libtsp-V2R7a $";

#include <libtsp.h>
#include <libtsp/nucleus.h>
#include <libtsp/AFpar.h>
#include <libtsp/UTtypes.h>

#define MINV(a, b)	(((a) < (b)) ? (a) : (b))

#define NBBUF	256

int
AFwriteHead (fp, offs, buf, size, Nelem, Swapb)

     FILE * fp;
     long int offs;
     const void *buf;
     int size;
     int Nelem;
     int Swapb;

{
  int N, Nb, Nbt;
  static int Hbo = DS_UNDEF;
  uint1_t *p;
  union {		/* Force stringent alignment */
    double8_t b8[NBBUF/8];
    uint4_t b4[NBBUF/4];
    uint2_t b2[NBBUF/2];
    uint1_t b1[NBBUF];
  } Ubuf;

/* Determine if the data should be swapped */
  if (size != 1) {
    switch (Swapb) {
    case DS_EB:
    case DS_EL:
      if (Hbo == DS_UNDEF)
	Hbo = UTbyteOrder ();
      if (Hbo == Swapb)
	Swapb = DS_NATIVE;
      else
	Swapb = DS_SWAP;
      break;
    case DS_SWAP:
    case DS_NATIVE:
      break;
    default:
      UThalt ("AFreadHead: Invalid byte swap code");
    }
  }
  else
    Swapb = DS_NATIVE;

/* Write the data with optional swapping */
  Nbt = Nelem * size;	/* Nelem is decremented later */
  if (Swapb != DS_SWAP)

    /* Non-swapped data is written directly from the input buffer */
    FLwriteFile (fp, offs, buf, (size_t) size, (size_t) Nelem);

  else {

    /* Swapped data passes through a temporary buffer */
    p = (uint1_t *) buf;
    while (Nelem > 0) {

      N = MINV (NBBUF / size, Nelem);
      VRswapBytes (p, Ubuf.b1, (size_t) size, N);
      FLwriteFile (fp, offs, Ubuf.b1, (size_t) size, (size_t) N);

      Nb = N * size;
      offs += Nb;
      p += Nb;
      Nelem = Nelem - N;
    }
  }

  return Nbt;
}
