/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  AFILE *AFsetAFp (FILE *fp, int Op, int Ftype, int Format, int Swapb,
                   double ScaleF, long int Nchan, long int Start,
                   long int Ldata, long int Nsamp)

Purpose:
  Set up the parameters in an audio file structure

Description:
  This routine checks the input parameters and puts them into a new audio file
  structure.  The data size is checked against the data type and adjusted
  downward (with a warning message) if it does not correspond to an integral
  number of samples.

Parameters:
  <-  AFILE *AFp
      Audio file pointer for the audio file.  This routine allocates the space
      for this structure.
   -> int Op
      Operation for file; AF_WO or AF_RO
   -> int Ftype
      File type; FT_NH, FT_AFSP, FT_SUN, FT_WAVE, etc.
   -> int Format
      Data format; FD_MULAW8, FD_ALAW8, FD_UINT8, FD_INT8, FD_INT16,
      FD_FLOAT32, or FD_TEXT
   -> int Swabp
      Byte swap flag:
      DS_EB     - File data is in big-endian byte order
      DS_EL     - FIle data is in little-endian byte order
      DS_NATIVE - File data is in native byte order
      DS_SWAP   - File data is byte-swapped
   -> double ScaleF
      Scale factor applied to the data
   -> long int Nchan
      Number of interleaved channels
   -> long int Start
      Byte offset to the start of the data in the file
   -> long int Ldata
      Byte count for the data (must be zero for a write operation)
   -> long int Nsamp
      Number of samples (must be zero for a write operation)

Author / revision:
  P. Kabal  Copyright (C) 1996
  $Revision: 1.14 $  $Date: 1996/08/14 18:04:13 $

-------------------------------------------------------------------------*/

static char rcsid [] = "$Id: AFsetAFp.c 1.14 1996/08/14 libtsp-V2R7a $";

#include <stdio.h>
#include <libtsp.h>
#include <libtsp/nucleus.h>
#include <libtsp/AFpar.h>

static const int Lenb[NFD] = {
  0,
  FDL_MULAW8,
  FDL_ALAW8,
  FDL_UINT8,
  FDL_INT8,
  FDL_INT16,
  FDL_FLOAT32,
  FDL_TEXT
};

AFILE *
AFsetAFp (fp, Op, Ftype, Format, Swapb, ScaleF, Nchan, Start, Ldata, Nsamp)

     FILE *fp;
     int Op;
     int Ftype;
     int Format;
     int Swapb;
     double ScaleF;
     long int Nchan;
     long int Start;
     long int Ldata;
     long int Nsamp;

{
  AFILE *AFp;
  int Lw;

/* Check the parameters */
  if (Op != FO_RO && Op != FO_WO)
    UThalt ("AFsetAFp: Invalid file operation code");
  if (Ftype <= 0 || Ftype >= NFT)
    UThalt ("AFsetAFp: Invalid file type");
  if (Format <= 0 || Format >= NFD)
    UThalt ("AFsetAFp: Invalid data format code");

  if (Start < 0)
    UThalt ("AFsetAFp: Invalid data offset value");
  if (Ldata < 0)
    UThalt ("AFsetAFp: Invalid data length");
  if (Op == FO_WO && Ldata != 0)
    UThalt ("AFsetAFp: Data length must be zero");

/* Check data consistency */
  Lw = Lenb[Format];
  if (Lw != 0) {
    if ((Ldata % Lw) != 0) {
      UTwarn ("AFsetAFp - Non-integral number of samples");
      Ldata = Lw * (Ldata / Lw);
    }
    if (Ldata != Lw * Nsamp)
      UThalt ("AFsetAFp: No. samples incompatible with file size");
  }
  if (Nsamp % Nchan != 0)
    UTwarn ("AFsetAFp - No. samples not a multiple of no. channels");

  if (Lw <= 1)
    Swapb = DS_NATIVE;

  switch (Swapb) {
  case DS_EL:
  case DS_EB:
    if (UTbyteOrder () == Swapb)
      Swapb = DS_NATIVE;
    else
      Swapb = DS_SWAP;
    break;
  case DS_SWAP:
  case DS_NATIVE:
    break;
  default:
    UThalt ("AFsetAFp: Invalid byte swap code");
  }

/* Set the parameters for file access */
  AFp = (AFILE *) UTmalloc (sizeof (AFILE));
  AFp->fp = fp;
  AFp->Op = Op;
  AFp->Ftype = Ftype;
  AFp->Format = Format;
  AFp->Swapb = Swapb;
  AFp->ScaleF = ScaleF;
  AFp->Nchan = Nchan;
  AFp->Start = Start;
  AFp->End = Start + Ldata;
  AFp->Isamp = 0L;
  AFp->Nsamp = Nsamp;
  AFp->Novld = 0L;

  return AFp;
}
