/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  char *AFgenHinfo (double Sfreq, int *Nchar)
  void AFresHinfo (void)

Purpose:
  Form the audio file header information string
  Reset the audio file header information string

Description:
  This returns the header information string in a form suitable for writing
  directly to an AFsp file header.  The header string is formed from a
  combination of the standard header information and the user supplied
  information string.  There are three cases.
    - User header information not specified.  The standard header information
      is generated.
    - User header information string starts with a newline.  The standard
      header information is generated and then the user header information
      string is appended.
    - User header information string does not start with a newline.  Only the
      user header information string is used.

  Standard Header Information:
    date:1994/01/25 19:19:39 UTC    date
    sample_rate:8012.5              sampling frequency (only if non-integer)
    user:kabal@aldebaran            user
    program:CopyAudio               program name (if set via routine UTsetProg)

Parameters:
  <-  char *AFgenHinfo
      Pointer to the header information string.  This string is Nchar
      characters long.  It is in a form that can be directly written to the
      header, i.e. newline characters have been changed to null characters.
      This string is allocated by this routine.  The space occupied can be
      freed up by invoking this routine with an empty string.
   -> double Sfreq
      Sampling frequency.  If this value is non-integer, a sample_rate record
      is put in the standard header information string.
  <-  int *Nchar
      Number of characters in the header information string

Author / revision:
  P. Kabal  Copyright (C) 1996
  $Revision: 1.8 $  $Date: 1996/05/31 12:53:56 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: AFgenHinfo.c 1.8 1996/05/31 libtsp-V2R7a $";

#include <math.h>	/* floor */
#include <stdio.h>
#include <string.h>
#include <libtsp.h>
#include <libtsp/nucleus.h>
#include <libtsp/AUpar.h>

void
AFresHinfo ();

static char *Hinfo = NULL;

char *
AFgenHinfo (Sfreq, Nchar)

     double Sfreq;
     int *Nchar;

{
  char Sinfo[320];
  char *Uinfo;
  char *p;
  int ns, nu, i, k;

  Uinfo = AFgetHinfo ();
  if (Uinfo == NULL || Uinfo[0] == '\n') {

 /* Generate the standard header information */
    sprintf (Sinfo, "date:%.40s\n", UTdate(3));
    ns = strlen (Sinfo);	/* Some versions of sprintf return a pointer */
    if (Sfreq != floor (Sfreq)) {
      sprintf (&Sinfo[ns], "sample_rate:%.7g\n", Sfreq);
      ns += strlen (&Sinfo[ns]);
    }
    p = UTuserName ();
    if (*p != '\0') {
      sprintf (&Sinfo[ns], "user:%.40s\n", p);
      ns += strlen (&Sinfo[ns]);
    }
    p = UTgetProg ();
    if (*p != '\0') {
      sprintf (&Sinfo[ns], "program:%.40s", p);
      ns += strlen (&Sinfo[ns]);
    }
  }
  else {
    ns = 0;
    Sinfo[0] = '\0';
  }
  
 /* Add the user specified header information */
  if (Uinfo != NULL)
    nu = strlen (Uinfo);
  else
    nu = 0;
  Hinfo = (char *) UTrealloc (Hinfo, nu + ns + 1);
  strcpy (Hinfo, Sinfo);
  if (nu > 0)
    strcpy (&Hinfo[ns], Uinfo);

/* Change newlines to nulls (unless escaped) */
  for (i = 0, k = 0; Hinfo[i] != '\0'; ++i, ++k) {
    if (Hinfo[i] == '\\' && Hinfo[i+1] == '\n') {
      Hinfo[k] = '\n';
      ++i;
    }
    else if (Hinfo[i] == '\n')
      Hinfo[k] = '\0';
    else
      Hinfo[k] = Hinfo[i];
  }
  Hinfo[k] = '\0';

/* Return a pointer to the header information */
  *Nchar = k;
  return Hinfo;
}

/* Reset the audio file header information string */
void
AFresHinfo ()

{
  AFsetHinfo (NULL);		/* Reset to standard info only */
  UTfree ((void *) Hinfo);	/* Recover the allocated space */
  Hinfo = NULL;
}
