/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  void AFsetHinfo (const char String[])
  char *AFgetHinfo (void)

Purpose:
  Set the audio file header information string
  Get the audio file header information string

Description:
  This routine sets the information string to be written to an AFsp audio file
  header.  This file format allows for an arbitrary information string to be
  written at the end of the header.  By default the header information string
  gives date and user information.  This routine allows the user to specify an
  information string that is used in addition to, or in place of, the standard
  header information string.  This routine must be called before the audio
  file to be created is first opened using AFopenWrite.  AFopenWrite resets
  the header string to a NULL string.

  Standard Header Information: 
    date:1994/01/25 19:19:39 UTC    date
    sample_rate:8012.5              sampling frequency (if non-integer)
    user:kabal@aldebaran            user
    program:CopyAudio               program name (set by UTsetProg)

  Additional structured information records should follow this format; a named
  field terminated by a colon followed by numeric data or a character string.
  Comments can follow as unstructured information.  For the purpose of this
  routine, records are separated by newline characters. The last record need
  not be terminated by a newline character.  When written to the header, the
  newline characters are replaced by nulls.  To place a newline character
  into the header without having it serve as a record separator, escape the
  newline character by preceding it with a backslash character, i.e. in a
  C-language string an escaped newline character would appear as "\\\n".

  If the input header information string is a NULL pointer, the standard
  information string is used.  If the first character of the header information
  string is a newline character, the the input header information string is
  added to the existing information string (the standard information string
  and/or any previously specified header strings).  Otherwise the input header
  information string replaces the standard information string.  The following
  examples illustrate the behaviour.

  - AFsetHinfo (NULL)
      Reset, use only the standard information string
  - AFsetHinfo ("")
      No header information string
  - AFsetHinfo ("<info>")
      Replace existing information string with <info>, do not use the standard
      information string
  - AFsetHinfo ("\n<info>")
      Add <info> to the existing information string, if the overall information
      string begins with a newline character, the overall information string
      appears in the header after the standard information string

  The procedure AFgetHinfo is used internally by the audio file routines to
  retrieve the user supplied header information string.

Parameters:
  AFsetHinfo:
   -> const char String[]
      String containing the user supplied header information
  AFgetHinfo:
  <-  char *AFgetUinfo
      Pointer to the header information string.  A NULL pointer indicates that
      no information string has been set.

Author / revision:
  P. Kabal  Copyright (C) 1996
  $Revision: 1.27 $  $Date: 1996/05/31 12:50:13 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: AFsetHinfo.c 1.27 1996/05/31 libtsp-V2R7a $";

#include <string.h>
#include <libtsp.h>
#include <libtsp/AUpar.h>

static char *Hinfo = NULL;

void
AFsetHinfo (String)

     const char String[];

{
  int Ni;
  static int Ninfo = 0;

  if (String == NULL) {
    UTfree ((void *) Hinfo);
    Hinfo = NULL;
    Ninfo = 0;
  }
  else {

/* Copy the header information string to an allocated buffer */
    if (String[0] == '\n')
      Ni = Ninfo;
    else
      Ni = 0;
    Ninfo = Ni + strlen (String);
    if (Ninfo > AFsp_MAXINFO) {
      UTwarn ("AFsetHinfo - Header information string too long");
      Ninfo = AFsp_MAXINFO;
    }

    /* Allocate the buffer */
    if (Ni == 0) {
      UTfree ((void *) Hinfo);
      Hinfo = (char *) UTmalloc (Ninfo+1);
    }
    else
      Hinfo = (char *) UTrealloc (Hinfo, Ninfo+1);

    /* Copy the new header info to the buffer */
    strncpy (&Hinfo[Ni], String, (size_t) (Ninfo-Ni));
    Hinfo[Ninfo] = '\0';
  }
}

char *
AFgetHinfo ()

{
  if (Hinfo != NULL)
    return Hinfo;
  else
    return NULL;
}
