*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     DOUBLE PRECISION FUNCTION TRESPF (T, F, FRESP, NVAL)
*
*
* Purpose:
*     Evaluate a time response, given freq. response values
*
*
* Description:
*     This function returns the value of a time response at a given
*     time.  The time response function is calculated as the Fourier
*     transform of a symmetrical real-valued frequency response.  The
*     corresponding time response is real and symmetrical about zero.
*
*     The frequency response is specified in tabular form.  A piece-
*     wise monotonic cubic interpolation is used between the given
*     values.  The frequency response is specified for non-negative
*     frequencies.  The cubic interpolant has zero slope at the first
*     and last points specified.
*
*
* Parameters:
* D <-  TRESPF - Returned function value (double precision)
*
* D ->  T      - Time value at which the time response is to be
*                evaluated (double precision)
* R ->  F      - Array of frequency values.  These values must be in
*                increasing order, and have non-negative values.
* R ->  FRESP  - Array of frequency response values.  Value FRESP(i) is
*                the frequency response at frequency F(i).
* I ->  NVAL   - Number of frequency response values specified.  NVAL
*                must be at least two.
*
*
* Routines required:
*     CCIINT - Evaluate the integral of a cubic
*     CCPINT - Evaluate a cubic times cosine integral
*     DMCINM - Calculate a derivative for a cubic at a middle point
*     HALT   - Print an error message, stop with error status set
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/25 05:35:53 $
*
*
*-----------------------------------------------------------------------

      DOUBLE PRECISION FUNCTION TRESPF (T, F, FRESP, NVAL)


      DOUBLE PRECISION PI2
      PARAMETER (PI2=6.283 185 307 179 586 476 9D0)

      INTEGER NVAL
      INTEGER I

      REAL F(NVAL),FRESP(NVAL)
      REAL DMCINM

      DOUBLE PRECISION T
      DOUBLE PRECISION A,DL,DH,FL,FH,VL,VH,SUM
      DOUBLE PRECISION CCIINT,CCPINT


* Error checks
      IF (NVAL.LE.1)
     -  CALL HALT('TRESPF - Error, too few data values')
      IF (F(1).LT.0.0)
     -  CALL HALT('TRESPF - Error, negative frequency')

* Loop over intervals
      A=PI2*T
      FL=F(1)
      VL=FRESP(1)
      DL=0.0
      SUM=0.0
      DO 100 I=1,NVAL-1

        IF (F(I).GE.F(I+1))
     -    CALL HALT('TRESPF - Error, Frequency values not increasing')

* Calculate the slopes at the end of the interval
        IF (I.EQ.NVAL-1) THEN
          DH=0.
        ELSE
          DH=DMCINM(F(I+1)-F(I),F(I+2)-F(I+1),
     -              FRESP(I+1)-FRESP(I),FRESP(I+2)-FRESP(I+1))
        END IF

        FH=F(I+1)
        VH=FRESP(I)
        IF (A.EQ.0.0) THEN

* If a=0, the formula gives 0/0, and we have to resort to a direct
* evaluation of the integral without the cosine term.
          SUM=SUM + CCIINT(FL,FH,VL,VH,DL,DH)

        ELSE

* Evaluate the integral in the interval [f1,f2]
*
*   f2                          x2
*  Int P(f) cos(a f) df = 1/a  Int  P(x/a) cos(x) dx, where a=2 Pi t,
*   f1                          x1
*
* x1=a f1, x2=a f2.
* Let G(x)=P(x/a).
* G(x1)=P(f1),  G(x2)=P(f2),  G'(x1)=P'(f1)/a,  G'(x2)=P'(f2)/a.
          SUM=SUM + CCPINT(A*FL,A*FH,VL,VH,DL/A,DH/A)
        END IF

        FL=FH
        VL=VH
        DL=DH

 100  CONTINUE

* Return the value of the integral
      IF (A.EQ.0.0) THEN
        TRESPF=2.*SUM
      ELSE
        TRESPF=2.*SUM/A
      END IF


      RETURN

      END
