*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE TRCHEB (ACOF, NCOF, C, D, BCOF)
*
*
* Purpose:
*     Transform Chebyshev polynomial coefficients
*
*
* Description:
*     The inputs to this routine are a set of coefficients for an
*     expansion in Chebyshev polynomials in the variable x.  This
*     routine finds the corresponding set of coefficients in the
*     variable y, where y=cx+d.
*
*     The expansions in Chebyshev polynomials are
*
*       N-1               N-1
*       SUM a(n) T(n,x) = SUM b(n) T(n,cx+d) ,  where
*       n=0               n=0
*
*       T(n+1,x) = 2x T(n,x) - T(n-1,x)  with initial conditions
*                                        T(0,x)=1, T(1,x)=x.
*
*
* Parameters:
* D ->  ACOF   - Array of double precision coefficients
* I ->  NCOF   - Number of elements in ACOF and BCOF (at most 128)
* D ->  C      - Transformation scaling variable (double precision).
*                This value must be nonzero.
* D ->  D      - Transformation offset variable (double precision)
* D <-  BCOF   - Array of double precision coefficients
*
*
* Routines required:
*     None
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/25 05:35:54 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE TRCHEB (ACOF, NCOF, C, D, BCOF)


      INTEGER NCFMX
      PARAMETER (NCFMX=128)

      INTEGER NCOF
      INTEGER K,I

      DOUBLE PRECISION ACOF(0:*),C,D,BCOF(0:*)
      DOUBLE PRECISION BNP(0:NCFMX-2),BNPP(0:NCFMX-3),P,Q


* Define the partial sum with k terms as
*
*            N-1
*   S(k,x) = SUM a(n) T(n-N+k,x) .
*           n=N-k
*
* The partial sum can be calculated recursively,
*
*   S(k,x) = 2x S(k-1,x) - S(k-2,x) - x a(N-k+1) + a(N-k) ,
*            k=1,...,N   with initial conditions S(0,x) = S(-1,x) = 0.
*
* This difference equation describes a linear system with inputs
* x a(N-k+1) and a(N-k).  The first input is a scaled shift of the
* second input.  Consider the simpler difference equation
*
*   S'(k,x) = 2x S'(k-1,x) - S'(k-2,x) + a(N-k) ,
*             k=1,...,N  with initial conditions
*             S'(0,x) = S'(-1,x) = 0.
*
* Then S'(k,x) is the output for the input a(N-k) and x S'(k-1,x) is
* the output for the input x a(N-k+1), and the overall partial sum is
*
*   S(k,x) = S'(k,x) - x S'(k-1,x).
*
* The sum S(N,x) is calculated as follows.
* (1) S'(k,x) = 2x S'(k-1,x) - S'(k-2,x) + a(N-k) ,  for k=1,...,N-1.
* (2) For the last step,
*     S(N,x) = S'(N,x) - x S'(N-1,x)
*            = x S'(N-1,x) - S'(N-2,x) + a(N-k) .
*     This recursion is the same as the general step (1), but with
*     2x replaced by x.
*
* Define an equivalent representation in terms of the Chebyshev
* polynomials in y,
*
*             k-1
*   S'(k,y) = SUM b(n,k) T'(n,y) .
*             n=0
*
* S'(k-1,y) and S'(k-2,y) have similar expressions.  A recursion
* for the coefficients b(n,k) can be developed from the equation
*
*   S'(k,y) - 2x S'(k-1,y) + S'(k-2,y) - a(N-k) = 0 ,
*
*   k-1
*   SUM [ b(n,k) - 2x b(n,k-1) + b(n,k-2) ] T'(n,y) - a(N-k) = 0 .
*   n=0
*
* Noting that x=py+q, and 2y T'(n,y)=T'(n+1,y)+T'(n-1,y), the
* expansion becomes
*
*   k-1
*   SUM [ b(n,k) - p b(n-1,k-1) - p b(n+1,k-1) - 2q b(n,k-1)
*   n=0
*         + b(n,k-2) ] T'(n,y) - p b(0,k-1) T'(1,y) - a(N-k) = 0 .
*
* The coefficients of the Chebyshev polynomials must be zero, giving
* the recursion
*
*   b(n,k) = 2q b(n,k-1) + p [b(n-1,k-1) + b(n+1,k-1)]
*            - b(n,k-2) - p b(0,k-1) D(1,n) + a(N-k) D(0,n) ,
*
* where D(n,m)=1 if n=m, and 0 otherwise.  This recursion is used to
* update the coefficients of the Chebyshev polynomials T'(n,y)
* to compute S'(N-1,x). The last step uses x instead of 2x to calculate
* S(N,x).


* Define the parameters of the inverse transformation
      P=1D0/C
      Q=-D/C

* Find the coefficients of the transformed Chebyshev series
*  BCOF(n) - b(n,k)
*  BNP(n)  - b(n,k-1)
*  BNPP(n) - b(n,k-2)
*  ACOF(n) - a(n)
      DO 400 K=1,NCOF
        IF (K.EQ.NCOF) THEN
          P=0.5D0*P
          Q=0.5D0*Q
        END IF

        DO 220 I=0,K-3
          BNPP(I)=BNP(I)
 220    CONTINUE

        DO 240 I=0,K-2
          BNP(I)=BCOF(I)
          BCOF(I)=2D0*Q*BNP(I)
 240    CONTINUE
        BCOF(K-1)=0D0
        BCOF(0)=BCOF(0)+ACOF(NCOF-K)

        IF (K.GE.2) BCOF(1)=BCOF(1)+P*BNP(0)
        DO 260 I=0,K-3
          BCOF(I)=BCOF(I)-BNPP(I)+P*BNP(I+1)
 260    CONTINUE

        DO 280 I=1,K-1
          BCOF(I)=BCOF(I)+P*BNP(I-1)
 280    CONTINUE

 400  CONTINUE


      RETURN

      END
