*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE REMEXC (AD, X, Y, NEXT,
*                        NGRID, XGRID, DES, WT, DLIML, DLIMU,
*                        DEV, IEXT, LIM, NCHNGE)
*
*
* Purpose:
*     Extremal search for filter design
*
*
* Description:
*     This routine finds the locations of the extrema of the error
*     function.
*
*
* Parameters:
* D ->  AD     - Double precision array of Lagrange interpolation
*                coefficients
* D ->  X      - Double precision array of abscissa values for the
*                Lagrange interpolation
* D ->  Y      - Double precision array of ordinate values for the
*                Lagrange interpolation
* I ->  NEXT   - Number of interpolation points
* I ->  NGRID  - Number of elements in each of XGRID, DES, WT, DLIML,
*                and DLIMU
* R ->  XGRID  - Array of abscissa values (grid points).  These values
*                must be in decreasing order and lie in the range
*                [-1,+1].
* R ->  DES    - Desired values on the grid
* R ->  WT     - Weight values on the grid, WT(i) > 0
* R ->  DLIML  - Lower constraint values on the grid,
*                DLIML(i) <= DES(i)
* R ->  DLIMU  - Upper constraint values on the grid,
*                DLIMU(i) >= DES(i)
* R ->  DEV    - Deviation at the design points
* I <-> IEXT   - Indices of the extremal points (N values).  These
*                values are updated by this routine.
* I <-> LIM    - Type of extremum (N values).  These values are updated
*                by this routine.  LIM(j) takes on one of the following
*                values,
*                -2 - Lower constraint limit
*                -1 - Lower ripple
*                +1 - Upper ripple
*                +2 - Upper constraint limit
* I <-  NCHNGE - Number of extrema changed.  A zero value signals that
*                the Remez exchange algorithm has converged.
*
*
* Routines required:
*     ERRMAX - Find a local extremum (Remez exchange algorithm)
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/25 05:32:29 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE REMEXC (AD, X, Y, NEXT,
     -                   NGRID, XGRID, DES, WT, DLIML, DLIMU,
     -                   DEV, IEXT, LIM, NCHNGE)


      INTEGER LLIM,LDEV,UDEV,ULIM
      PARAMETER (LLIM=-2,LDEV=-1,UDEV=+1,ULIM=+2)

      INTEGER NEXT,NGRID,IEXT(NEXT),LIM(NEXT),NCHNGE
      INTEGER L1,LN,LLOW,I,LUP,L,LST,LFN,LEXT,IS,LIM0,L0,
     -        LIMNP1,LNP1

      REAL XGRID(NGRID),DES(NGRID),WT(NGRID),
     -     DLIML(NGRID),DLIMU(NGRID),DEV
      REAL ERR,ERR1,ERRN,ERR01

      DOUBLE PRECISION AD(NEXT),X(NEXT),Y(NEXT)


      NCHNGE=0
      L1=IEXT(1)
      LN=IEXT(NEXT)
      LLOW=0


****************
* Search for the extremal frequencies of the best approximation

      DO 300 I=1,NEXT

        IF (I.EQ.NEXT) THEN
          LUP=NGRID+1
        ELSE
          LUP=IEXT(I+1)
        END IF
        L=IEXT(I)
        ERR=DEV

* Look for a local maximum by increasing the frequency
        LST=L+1
        LFN=LUP-1
        IS=+1
        CALL ERRMAX(LST,LFN,IS,LEXT,ERR,DEV,LIM(I),
     -              AD,X,Y,NEXT,
     -              NGRID,XGRID,DES,WT,DLIML,DLIMU)
        IF (LEXT.GE.0) GO TO 260


* Look for a local maximum by decreasing the frequency
        LST=L-1
        LFN=LLOW+1
        IS=-1
        CALL ERRMAX(LST,LFN,IS,LEXT,ERR,DEV,LIM(I),
     -              AD,X,Y,NEXT,
     -              NGRID,XGRID,DES,WT,DLIML,DLIMU)
        IF (LEXT.GE.0) GO TO 240


* Keep looking for a local maximum, decreasing the frequency
        IF (NCHNGE.EQ.0) THEN

          LST=L-2
          LFN=LLOW+1
          IS=-2
          CALL ERRMAX(LST,LFN,IS,LEXT,ERR,DEV,LIM(I),
     -                AD,X,Y,NEXT,
     -                NGRID,XGRID,DES,WT,DLIML,DLIMU)
          IF (LEXT.GE.0) GO TO 240


* Keep looking for a local maximum, increasing the frequency
          LST=L+2
          LFN=LUP-1
          IS=+2
          CALL ERRMAX(LST,LFN,IS,LEXT,ERR,DEV,LIM(I),
     -                AD,X,Y,NEXT,
     -                NGRID,XGRID,DES,WT,DLIML,DLIMU)
          IF (LEXT.GE.0) GO TO 260

        END IF

* No new local maximum found
        LLOW=L
        GO TO 280


* A local maximum found at a lower frequency
 240    LLOW=L
        IEXT(I)=LEXT
        NCHNGE=NCHNGE+1
        GO TO 280


* A local maximum found at a higher frequency
 260    LLOW=LEXT
        IEXT(I)=LEXT
        NCHNGE=NCHNGE+1
        GO TO 280

* Save the error for the first extremum
 280    IF (I.EQ.1) ERR1=ERR

 300  CONTINUE

* Save the error for the last extremum
      ERRN=ERR

****************

* Endpoints search

* Search below the lower endpoint for a local maximum
* (Look for an error larger than the error found at the
*  upper endpoint)
      ERR=1.00001*ERRN
      LST=1
      LFN=MIN(L1,IEXT(1))-1
      IS=+2
      IF (LIM(1).EQ.LDEV .OR. LIM(1).EQ.LLIM) THEN
        LIM0=UDEV
      ELSE
        LIM0=LDEV
      END IF
      CALL ERRMAX(LST,LFN,IS,L0,ERR,DEV,LIM0,
     -            AD,X,Y,NEXT,
     -            NGRID,XGRID,DES,WT,DLIML,DLIMU)
      IF (L0.GE.0) THEN
        ERR01=MAX(ERR1,ERR)
      ELSE
        ERR01=ERR1
      END IF

* Search above the upper endpoint for a local maximum
* (Look for an error larger than the error found at or below
*  the lower endpoint)
      ERR=1.00001*ERR01
      LST=NGRID
      LFN=MAX(LN,IEXT(NEXT))+1
      IS=-2
      IF (LIM(NEXT).EQ.LDEV .OR. LIM(NEXT).EQ.LLIM) THEN
        LIMNP1=UDEV
      ELSE
        LIMNP1=LDEV
      END IF
      CALL ERRMAX(LST,LFN,IS,LNP1,ERR,DEV,LIMNP1,
     -            AD,X,Y,NEXT,
     -            NGRID,XGRID,DES,WT,DLIML,DLIMU)
      IF (LNP1.GE.0) THEN

* Found a local maximum above the former upper endpoint
        DO 420 I=1,NEXT-1
          IEXT(I)=IEXT(I+1)
          LIM(I)=LIM(I+1)
 420    CONTINUE
        IEXT(NEXT)=LNP1
        LIM(NEXT)=LIMNP1
        NCHNGE=NCHNGE+1

      ELSE IF (L0.GE.0) THEN

* Found a local maximum below the former lower endpoint
        DO 440 I=NEXT-1,1,-1
          IEXT(I+1)=IEXT(I)
          LIM(I+1)=LIM(I)
 440    CONTINUE
        IEXT(1)=L0
        LIM(1)=LIM0
        NCHNGE=NCHNGE+1

      END IF


      RETURN

      END
