*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE REMDEV (AD, NEXT, NGRID, DES, WT, DLIML, DLIMU,
*                        IEXT, LIM, NCOF, DEVS, IER)
*
*
* Purpose:
*     Calculate the deviation for a minimax approximation
*
*
* Description:
*     This routine calculates the deviation for a minimax
*     approximation.  The input is a set of desired values, weights
*     and limits at NEXT points.  This routine solves for the deviation
*     that allows for a NCOF term polynomial to alternate about
*     the desired values.  In the general case, the polynomial has
*     NCOF=NEXT-1 terms.  For the special case NEXT <= NCOF, the
*     polynomial has NEXT terms and can match the desired values at
*     all NEXT points (DEVS=0).
*
*
* Parameters:
* D ->  AD     - Double precision array of Lagrange interpolation
*                coefficients (NEXT values)
* I ->  NEXT   - Number of extremal values
* I ->  NGRID  - Number of elements in each of DES, WT, DLIML, and
*                DLIMU.  NGRID must be at least as large as NEXT.
* R ->  DES    - Desired values
* R ->  WT     - Weight values, WT(i) > 0
* R ->  DLIML  - Lower constraint values, DLIML(i) <= DES(i)
* R ->  DLIMU  - Upper constraint values, DLIMU(i) >= DES(i)
* I ->  IEXT   - Indices of the extremal points (NEXT values)
* I ->  LIM    - Type of extremum (NEXT values).  LIM(i) takes on one
*                of the following values,
*                -2 - Lower constraint limit
*                -1 - Lower ripple
*                +1 - Upper ripple
*                +2 - Upper constraint limit
* I ->  NCOF   - Number of terms for the polynomial fit
* R <-  DEVS   - Deviation at the design points
* I <-  IER    - Error parameter.
*                0 - No error
*                3 - Constraints too tight for the given filter order
*                    (DEVS is set to 1E20)
*
*
* Routines required:
*     WARN   - Print a warning message on the standard error unit
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/25 05:32:30 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE REMDEV (AD, NEXT, NGRID, DES, WT, DLIML, DLIMU,
     -                   IEXT, LIM, NCOF, DEVS, IER)


      INTEGER LLIM,LDEV,UDEV,ULIM
      PARAMETER (LLIM=-2,LDEV=-1,UDEV=+1,ULIM=+2)

      INTEGER NEXT,NGRID,IEXT(NEXT),LIM(NEXT),NCOF,IER
      INTEGER I,L

      REAL DES(NGRID),WT(NGRID),DLIML(NGRID),DLIMU(NGRID),DEVS

      DOUBLE PRECISION AD(NEXT)
      DOUBLE PRECISION DNUM,DDEN


* Calculate the deviation:
* Given a set of NEXT distinct points, an NEXT-1 coefficient polynomial
* can give an error curve which exhibits equal amplitude alternations
* (in a weighted sense) about the desired value.

* In such a formulation, the free parameters are the NEXT-1
* coefficients and the deviation giving a total of NEXT linear
* equations.  The strategy is to solve for the unique value of the
* deviation which allows for this alternation of the error, without
* solving directly for the coefficients of the polynomial.  The case of
* NCOF >= NEXT gives zero deviation.

* The deviation is found using Cramer's rule.  The determinants in this
* formulation can be expressed in terms of the determinant of a
* Vandermonde matrix.  The expression for the deviation is expressed in
* terms of the Lagrange coefficients for an NEXT value interpolation.
      IF (NCOF.GE.NEXT) THEN
        IER=0
        DEVS=0.0

      ELSE

        DNUM=0D0
        DDEN=0D0
        DO 120 I=1,NCOF+1
          L=IEXT(I)
          IF (LIM(I).EQ.LDEV) THEN
            DNUM=DNUM+AD(I)*DES(L)
            DDEN=DDEN+AD(I)/WT(L)
          ELSE IF (LIM(I).EQ.UDEV) THEN
            DNUM=DNUM+AD(I)*DES(L)
            DDEN=DDEN-AD(I)/WT(L)
          ELSE IF (LIM(I).EQ.LLIM) THEN
            DNUM=DNUM+AD(I)*DLIML(L)
          ELSE
            DNUM=DNUM+AD(I)*DLIMU(L)
          END IF
 120    CONTINUE
        IF (DDEN.EQ.0D0) THEN
          CALL WARN('REMDEV - Constraints too tight')
          IER=3
          DEVS=1E20
        ELSE
          IER=0
          DEVS=DNUM/DDEN
        END IF

      END IF


      RETURN

      END
