*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE REMCOF (XS, XF, NEXT, X, Y, NCOF, ALPHA)
*
*
* Purpose:
*     Generate the sum of cosines coefficients for a filter
*
*
* Description:
*     Given the sample values for a polynomial, this routine returns
*     the coefficients of the sum of cosines.  The approximation is of
*     the form
*
*               Ncof-1
*        P(w) =  SUM  a(n) cos(wn) .
*                n=0
*
*     The input values are NEXT values of P(w),
*       y(i)=P(w(i)) and x(i)=cos(w(i)), i=1,...,NEXT.
*     These values are assumed to coincide with a polynomial
*     specified by Ncof coefficients.
*
*
* Parameters:
* R ->  XS     - Starting abscissa limit. This value is normally in the
*                interval [max(X(i)),+1].  Together with XF it defines
*                the interval over which the design was specified.
* R ->  XF     - Final abscissa limit.  This value is normally in the
*                interval [-1,min(X(i))].  Together with XS it defines
*                the interval over which the design was specified.
* I ->  NEXT   - Number of abscissa values.  The corresponding values
*                are assumed to lie on an NCOF term polynomial.
* D ->  X      - Array of NEXT abscissa values (-1 to +1) (double
*                precision).  These values must be in decreasing order.
* D ->  Y      - Array of NEXT ordinate values (double precision)
* I ->  NCOF   - Number of coefficients (maximum 128)
* D <-  ALPHA  - Returned double precision array of coefficients (NCOF
*                values)
*
*
* Routines required:
*     LGCOEF - Generate Lagrange interpolation coefficients
*     TRCHEB - Transform Chebyshev polynomial coefficients
*     VLGINT - Perform Lagrange interpolation
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/25 05:32:31 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE REMCOF (XS, XF, NEXT, X, Y, NCOF, ALPHA)


      INTEGER NCFMX
      REAL EPS,PI2,FSH
      PARAMETER (NCFMX=128,EPS=0.05)
      PARAMETER (PI2=6.283 185 307,FSH=1.0E-6)

      INTEGER NEXT,NCOF
      INTEGER M,L,K,I

      LOGICAL FULLB

      REAL XS,XF
      REAL XT

      DOUBLE PRECISION X(NEXT),Y(NEXT),ALPHA(0:NCOF-1)
      DOUBLE PRECISION DSUM,VLGINT,C,D
      DOUBLE PRECISION AD(NCFMX+1)
      DOUBLE PRECISION P(0:NCFMX-1),ACOF(0:NCFMX-1)


* If the band specifications do not extend from 0 to 1/2, the frequency
* response is not sampled at the edges.

* The full band samples are w(k) = 2 Pi k/M.
* For the partial band sampling, define the transformed (radian)
* frequency u which satisfies cos(u) = c cos(w) + d.
      C=0.5*(XF-XS)
      D=0.5*(XF+XS)
      FULLB=C.EQ.0D0 .OR. ((XS.GE.+1.0-EPS) .AND. (XF.LE.-1.0+EPS))

* Generate the Lagrange interpolation coefficients
      CALL LGCOEF(AD,X,NEXT)

* Evaluate the function using VLGINT at equally spaced intervals
      M=2*NCOF-1
      L=1
      DO 200 K=0,NCOF-1

        IF (.NOT.FULLB) THEN
          XT=C*COS(K*PI2/M)+D
        ELSE
          XT=COS(K*PI2/M)
        END IF

* If the abscissa value lies close to one of the grid values,
* use the corresponding ordinate value to avoid a divide by zero
 120    IF (XT.LE.X(L)-FSH .AND. L.LT.NEXT) THEN
          L=L+1
        GO TO 120
        END IF
        IF (XT.LT.X(L)+FSH .AND. XT.GT.X(L)-FSH) THEN
          P(K)=Y(L)
        ELSE
          P(K)=VLGINT(XT,AD,X,Y,NEXT)
        END IF

 200  CONTINUE

* Evaluate the inverse discrete Fourier transform to find the
* cosine series coefficients

* The frequency response is
*
*          N-1                 M-1
*   P(w) = SUM a(n) cos(wn)  = SUM b(n) exp(jwn)
*          n=0                 n=0
*
* where M = 2N-1,
*       b(0) = a(0),
*       b(n) = a(n)/2 for 0 < n < N
*       b(n) = b(M-n) for N <= n < M .
*
* Let P(k) = P(2 Pi k/M), the DFT of {b(0),b(1),...,b(M-1)}.
* P(k) = P(M-k).
* The inverse DFT gives
*
*          1  M-1
*   b(n) = -  SUM P(k) exp(j 2 Pi kn/M)
*          M  k=0
*
*          P(0)   1 N-1
*        = ---- + - SUM P(k) cos(2 Pi kn/M) .
*           M     M k=1
* Then
*          1           N-1
*   a(0) = - [P(0) + 2 SUM P(k)] ,
*          M           k=1
*
*          1            N-1
*   a(n) = - [ P(0) + 2 SUM P(k) cos(2 Pi kn/M) ],  0 < n < N .
*          M            k=1
*
      DSUM=0D0
      DO 320 K=1,NCOF-1
        DSUM=DSUM+P(K)
 320  CONTINUE
      ACOF(0)=( P(0) + 2D0*DSUM ) / M

      DO 360 I=1,NCOF-1
        DSUM=0D0
        DO 340 K=1,NCOF-1
          DSUM=DSUM+P(K)*COS(K*I*PI2/M)
 340    CONTINUE
        ACOF(I)=2D0*( P(0) + 2D0*DSUM ) / M
 360  CONTINUE


* The inverse discrete Fourier transform gives the coefficients
* a(n) of a Chebyshev series expansion,
*
*          N-1                N-1
*   P(x) = SUM a(n) cos(wn) = SUM a(n) T(n,x) .
*          n=0                n=0
*
* The Chebyshev polynomials are polynomials in x=cos(w), and
* satisfy the recurrence relationship,
*
*   T(n+1,x) = 2x T(n,x) - T(n-1,x), with T(0,x)=1 and T(1,x)=x.
*
* However what we really want are the coefficients of the expansion
* in y=cos(u),
*
*          N-1                N-1
*   P(u) = SUM b(n) cos(un) = SUM b(n) T(n,y) ,
*          n=0                n=0
*
* where T(n,y) satisfies the recursion,
*
*   T(n+1,y) = 2y T(n,y) - T(n-1,y) .
*
* Subroutine TRCHEB finds the coefficients b(n), given the coefficients
* a(n) and the transformation y=cx+d.

      IF (.NOT.FULLB) THEN

* Find the coefficients for partial band sampling
        CALL TRCHEB(ACOF,NCOF,C,D,ALPHA)

      ELSE

* Coefficients for full band sampling
        DO 420 I=0,NCOF-1
          ALPHA(I)=ACOF(I)
 420    CONTINUE

      END IF


      RETURN

      END
