*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE REDINT (NAMEFL, NTAP, IR, F, PSD, NVAL)
*
*
* Purpose:
*     Read interpolating filter specifications
*
*
* Description:
*     This subroutine issues prompts for and reads specifications for
*     an interpolating filter. If an error is detected, a message is
*     signalled and execution is halted.
*
*
* Parameters:
* C <-  NAMEFL - Character string specifying the coefficient file name.
*                NAMEFL may be all blanks.
* I <-  NTAP   - Filter length
* I <-  IR     - Interpolation ratio
* R <-  F      - Array of normalized frequencies.  The frequency values
*                are non-negative.  The normalizing frequency is the
*                frequency of the input sequence before interpolation.
*                The first NVAL(1) values must be in increasing order.
*                These are the frequencies of samples of the continuous
*                power spectrum.  The next NVAL(2) values are the
*                frequencies of the sinusoidal components.
* R <-  PSD    - Array of power spectral density values.  The value
*                PSD(i) is the power spectral density at frequency
*                F(i).  Each PSD(i) is non-negative.  The first NVAL(1)
*                values are samples of the continuous power spectrum.
*                The next NVAL(2) values are the powers of the
*                sinusoidal components at the corresponding frequency.
* I <-  NVAL   - Number of frequency and power spectral density values.
*                This is a two element array.  NVAL(1) gives the number
*                of samples of the continuous power spectrum (at least
*                2). NVAL(2) gives the number of sinusoidal components.
*
*
* Routines required:
*     HALT   - Print an error message, stop with error status set
*     ICEIL  - Calculate the ceiling function of a ratio of integers
*     KEYUPC - Match keyword strings (case insensitive)
*     LENNUL - Find the length of a null terminated string
*     RDCSTR - Separate comma/whitespace delimited substrings
*     RDIVA1 - Decode an integer value
*     RDLINE - Read a line of input from standard input
*     RDRVA1 - Decode a real value
*     WRTTY  - Write to standard output if input is from a terminal
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.4 $  $Date: 1993/02/03 03:46:44 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE REDINT (NAMEFL, NTAP, IR, F, PSD, NVAL)


      INTEGER NFMAX,NMAX,MXPT
      PARAMETER (NFMAX=399,NMAX=42,MXPT=200)

      CHARACTER*(*) NAMEFL
      CHARACTER*80 LINE,SLINE(3)

      INTEGER NTAP,IR,NVAL(2)
      INTEGER NCHR,ISRC,IER,N,NM,NCHRN,I,IT,K
      INTEGER LENNUL,ICEIL,KEYUPC

      REAL F(*),PSD(*)
      REAL FT(MXPT,2),PSDT(MXPT,2)
      REAL SFREQ,FRP

      CHARACTER*4 PSDTAB(2)
      DATA PSDTAB/'*PSD', 'C*OS'/


* Read the interpolating ratio
      CALL RDLINE('Interpolating ratio: ',LINE,NCHR,ISRC)
        IF (NCHR.EQ.0) CALL HALT('REDINT - Insufficient data')
      CALL RDIVA1(LINE,IR,IER)
      IF (IER.NE.0) CALL HALT('REDINT - Data error')
      IF (IR.LE.0) CALL HALT('REDINT - Invalid interpolating ratio')

* Read the number of coefficients
      CALL RDLINE('No. coeff, Input Sampling Rate: ',LINE,NCHR,ISRC)
        IF (NCHR.EQ.0) CALL HALT('REDINT - Insufficient data')
      CALL RDCSTR(LINE,1,2,SLINE,N)
        IF (N.EQ.0) CALL HALT('RDFIR - Invalid number of parameters')
      NCHR=LENNUL(SLINE(1))
      CALL RDIVA1(SLINE(1)(1:NCHR),NTAP,IER)
      IF (IER.NE.0) CALL HALT('REDINT - Data error')

* Find the length of the longest subfilter
* For NTAP odd, the length of the zero'th subfilter need not
* counted since it has known coefficients 0 ... 0 1 0 ... 0.
* (In this case reduce the subfilter length by 1)
      NM=ICEIL(NTAP,IR)
      IF (MOD(NTAP-1,2*IR).EQ.0) NM=NM-1
      IF (NTAP.LE.0)
     -  CALL HALT('REDINT - Invalid number of coefficients')
      IF (NTAP.GT.NFMAX .OR. NM.GT.NMAX)
     -  CALL HALT('REDINT - Too many coefficients')

* Decode the optional parameter
      SFREQ=1.0
      IF (N.EQ.2) THEN
        NCHR=LENNUL(SLINE(2))
        IF (NCHR.GT.0) THEN
          CALL RDRVA1(SLINE(2)(1:NCHR),SFREQ,IER)
          IF (IER.NE.0) CALL HALT('REDINT - Data error')
          IF (SFREQ.LE.0.0)
     -      CALL HALT('REDINT - Invalid sampling frequency')
        END IF
      END IF

* Read the coefficient file name
      CALL RDLINE('Coefficient file: ',NAMEFL,NCHRN,ISRC)
        IF (ISRC.LT.0) CALL HALT('REDINT - Insufficient data')

* Read the values of the power spectrum of the input process
      CALL WRTTY('Input signal power spectrum')
      NVAL(1)=0
      NVAL(2)=0
      FRP=-1.
      DO 200 I=1,MXPT+1
        CALL RDLINE('  Frequency, Value: ',LINE,NCHR,ISRC)
          IF (ISRC.LT.0 .OR. NCHR.EQ.0) GO TO 220
        IF (I.GT.MXPT)
     -    CALL HALT('REDINT - Too many specification records')
        CALL RDCSTR(LINE,2,3,SLINE,N)
          IF (N.LE.0)
     -      CALL HALT('REDINT - Invalid number of values')

* Distinguish between continuous psd and sinusoid components
* IT=1 continuous psd,  IT=2 sinusoidal component
        IF (N.EQ.3) THEN
          IT=KEYUPC(SLINE(3),PSDTAB,2)
            IF (IT.EQ.0)
     -        CALL HALT('REDINT - Invalid power spectrum identifier')
        ELSE
          IT=1
        END IF
        NVAL(IT)=NVAL(IT)+1
        K=NVAL(IT)

* Decode the power spectrum values
        CALL RDRVA1(SLINE(1),FT(K,IT),IER)
          IF (IER.NE.0) CALL HALT('REDINT - Data error')
        FT(K,IT)=FT(K,IT)/SFREQ
        CALL RDRVA1(SLINE(2),PSDT(K,IT),IER)
          IF (IER.NE.0) CALL HALT('REDINT - Data error')

* Error checks
        IF (FT(K,IT).LT.0.0)
     -    CALL HALT('REDINT - Invalid frequency')
        IF (PSDT(K,IT).LT.0.0)
     -    CALL HALT('REDINT - Negative power spectrum value')
        IF (IT.EQ.1) THEN
          IF (FT(K,1).LE.FRP)
     -      CALL HALT('REDINT - Improperly ordered frequency')
          FRP=FT(K,1)
        END IF

 200  CONTINUE


 220  IF (NVAL(1).EQ.0 .AND. NVAL(2).EQ.0)
     -  CALL HALT('REDINT - No power spectrum values specified')

* Copy the power spectral values into the output array
      DO 300 I=1,NVAL(1)
        F(I)=FT(I,1)
        PSD(I)=PSDT(I,1)
 300  CONTINUE
      DO 320 K=1,NVAL(2)
        F(I)=FT(K,2)
        PSD(I)=PSDT(K,2)
        I=I+1
 320  CONTINUE


      RETURN

      END
