*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE REDFIR (ITYPE, NTAP, NAMEFL, GRIDD, IBAND, NBANDS,
*                        FREQ, VAL, WEIGHT, VLIML, VLIMU)
*
*
* Purpose:
*     Read FIR filter specifications
*
*
* Description:
*     This subroutine issues prompts and then reads filter design
*     specifications.  A maximum of 200 frequency points may be
*     specified.  If an error is detected, a message is signalled and
*     execution is halted.
*
*
* Parameters:
* I ->  ITYPE  - Filter type coded as follows.
*                1 - Multiple passband/stopband filter
*                2 - Multiple passband/stopband filter
*                    (sin(X)/X compensation)
*                3 - Differentiator
*                4 - Hilbert transform filter
* I <-  NTAP   - Number of filter coefficients (maximum 256)
* C <-  NAMEFL - Character string specifying the coefficient file name.
*                NAMEFL may be all blank.
* R <-  GRIDD  - Grid density.  GRIDD is zero if unspecified, or equal
*                to -1, if MAXIMUM is specified.
* I <-  IBAND  - Frequency band specification index.  The array FREQ is
*                used to specify the frequency bands.
*                 Band    Lower Band Edge          Upper Band Edge
*                  1     FREQ(1)                  FREQ(IBAND(1))
*                  2     FREQ(IBAND(1)+1)         FREQ(IBAND(2))
*                 ...        ...                    ...
*                NBANDS  FREQ(IBAND(NBANDS-1)+1)  FREQ(IBAND(NBANDS))
*                VAL(i) and WEIGHT(i) specify the desired value and
*                weight at frequency FREQ(i).
* I <-  NBANDS - Number of bands specified
* R <-  FREQ   - Array of normalized frequencies.  These are in
*                increasing order.
* R <-  VAL    - Array of desired values
* R <-  WEIGHT - Array of desired weightings
* R <-  VLIML  - Array of lower constraints. If the constraints are not
*                specified for a frequency, the corresponding element
*                of VLIML is set to -1E20.
* R <-  VLIMU  - Array of upper constraints. If the constraints are not
*                specified for a frequency, the corresponding element
*                of VLIMU is set to +1E20.
*
*
* Routines required:
*     FIRCHK - Check FIR filter design specifications
*     HALT   - Print an error message, stop with error status set
*     KEYUPC - Match keyword strings (case insensitive)
*     LENBLK - Find the length of a blank terminated string
*     LENNUL - Find the length of a null terminated string
*     RDCSTR - Separate comma/whitespace delimited substrings
*     RDIVA1 - Decode an integer value
*     RDLINE - Read a line of input from standard input
*     RDRVA1 - Decode a real value
*     RDRVAL - Decode real values (variable number)
*     WRTTY  - Write to standard output if input is from a terminal
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.4 $  $Date: 1993/02/03 03:47:17 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE REDFIR (ITYPE, NTAP, NAMEFL, GRIDD, IBAND, NBANDS,
     -                  FREQ, VAL, WEIGHT, VLIML, VLIMU)


      INTEGER BPF,REC,DIF,HIL,NFMAX,MXPT
      PARAMETER (BPF=1,REC=2,DIF=3,HIL=4)
      PARAMETER (NFMAX=256,MXPT=200)

      CHARACTER*(*) NAMEFL
      CHARACTER*80 LINE,SLINE(3)
      CHARACTER*8 MAXTAB(1)
      CHARACTER*16 TRATAB(1)

      LOGICAL LIMS,LIMSP

      INTEGER ITYPE,NTAP,IBAND(*),NBANDS
      INTEGER NCHR,ISRC,IER,NCHRN,ITRANS,N,I,K
      INTEGER LENNUL,KEYUPC,LENBLK

      REAL GRIDD,FREQ(*),VAL(*),WEIGHT(*),VLIML(*),VLIMU(*)
      REAL RVAL(5)
      REAL SFREQ,FP

      DATA MAXTAB/'MAX*IMUM'/
      DATA TRATAB/'*TRANSITION BAND'/


* Read the number of coefficients
      CALL RDLINE('No. Coeff, Sampling rate: ',LINE,NCHR,ISRC)
        IF (NCHR.EQ.0) CALL HALT('REDFIR - Insufficient data')
      CALL RDCSTR(LINE,1,3,SLINE,N)
        IF (N.LE.0)
     -    CALL HALT('REDFIR - Invalid number of parameters')
      NCHR=LENNUL(SLINE(1))
      CALL RDIVA1(SLINE(1)(1:NCHR),NTAP,IER)
        IF (IER.NE.0) CALL HALT('REDFIR - Data error')
        IF (NTAP.LE.0)
     -    CALL HALT('REDFIR - Invalid number of coefficients')
        IF (NTAP.GT.NFMAX)
     -    CALL HALT('REDFIR - Too many coefficients')

* Decode the optional parameters SFREQ and GRIDD
      SFREQ=1.0
      IF (N.GE.2) THEN
        NCHR=LENNUL(SLINE(2))
        IF (NCHR.GT.0) THEN
          CALL RDRVA1(SLINE(2)(1:NCHR),SFREQ,IER)
          IF (IER.NE.0) CALL HALT('REDFIR - Data error')
          IF (SFREQ.LE.0)
     -      CALL HALT('REDFIR - Invalid sampling frequency')
        END IF
      END IF

      GRIDD=0.0
      IF (N.EQ.3) THEN
        NCHR=LENNUL(SLINE(3))
        IF (NCHR.GT.0) THEN
          N=KEYUPC(SLINE(3)(1:NCHR),MAXTAB,1)
          IF (N.EQ.1) THEN
            GRIDD=-1.
          ELSE
            CALL RDRVA1(SLINE(3)(1:NCHR),GRIDD,IER)
            IF (IER.NE.0) CALL HALT('REDFIR - Data error')
            IF (GRIDD.LE.0.0)
     -        CALL HALT('REDFIR - Invalid grid density')
          END IF
        END IF
      END IF

* Read the coefficient file name
      CALL RDLINE('Coefficient file: ',NAMEFL,NCHRN,ISRC)
        IF (ISRC.LT.0) CALL HALT('REDFIR - Insufficient data')

* Read the band specifications
*  ITRANS - -1 initially
*            0 mid-band
*           +1 transition
      FP=-1.
      NBANDS=0
      ITRANS=-1
      DO 140 I=1,MXPT+1
 120    IF (ITRANS.NE.0) THEN
          IF (ITYPE.NE.DIF) THEN
            WRITE (UNIT=LINE,FMT=2000) NBANDS+1
          ELSE
            WRITE (UNIT=LINE,FMT=2100) NBANDS+1
          END IF
          NC=LENBLK(LINE)
          CALL WRTTY(LINE(1:NC))
        END IF
        CALL RDLINE('  Band Data: ',LINE,NCHR,ISRC)
          IF (ISRC.LT.0 .OR.
     -        (NCHR.EQ.0 .AND. ITRANS.EQ.1)) GO TO 160

* Check for a transition band specification
        N=KEYUPC(LINE,TRATAB,1)
        IF (N.EQ.1) THEN
          IF (ITRANS.EQ.0) THEN
            NBANDS=NBANDS+1
            IBAND(NBANDS)=K
          END IF
          ITRANS=1
          GO TO 120
        ELSE

* Frequency point
          IF (I.GT.MXPT)
     -      CALL HALT('REDFIR - Too many specification records')
          K=I
          CALL RDRVAL(LINE,3,5,RVAL,N)
            IF (N.EQ.0 .OR. N.EQ.4)
     -        CALL HALT('REDFIR - Invalid number of parameters')
          FREQ(K)=RVAL(1)/SFREQ
          VAL(K)=RVAL(2)
          WEIGHT(K)=RVAL(3)
          LIMS=N.EQ.5
          IF (LIMS) THEN
            VLIML(K)=RVAL(4)
            VLIMU(K)=RVAL(5)
          ELSE
            VLIML(K)=-1E20
            VLIMU(K)=+1E20
          END IF

* Error checks
          CALL FIRCHK(FREQ(K),VAL(K),WEIGHT(K),VLIML(K),VLIMU(K),
     -                ITYPE,NTAP)
          IF (FREQ(K).LE.FP)
     -      CALL HALT('REDFIR - Frequency values not in '//
     -                'increasing order')
          IF (ITRANS.EQ.0 .AND. (LIMS.NEQV.LIMSP))
     -      CALL HALT('REDFIR - Constraint values must be '//
     -                'specified for all or no points in a band')

          ITRANS=0
          IF (FREQ(K).GE.0.5) GO TO 160
          FP=FREQ(K)
          LIMSP=LIMS
        END IF
 140  CONTINUE

 160  IF (ITRANS.EQ.0) THEN
        NBANDS=NBANDS+1
        IBAND(NBANDS)=K
      END IF

      IF (NBANDS.LE.0)
     -   CALL HALT('REDFIR - No frequency bands specified')


      RETURN

 2000 FORMAT (' Band',I3,': Frequency, Value, Weight[, Low, High]')
 2100 FORMAT (' Band',I3,': Frequency, Slope, Weight[, Low, High]')

      END
