*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE IMPULS (ICASE, ALPHA, H, NTAP)
*
*
* Purpose:
*     Find the impulse response for an FIR filter
*
*
* Description:
*     This routine finds the impulse response of a linear phase FIR
*     filter given the coefficients of a canonical response in terms of
*     a cosine expansion.  Four different cases are considered (NTAP
*     even or odd, and H symmetric or anti-symmetric).
*
*
* Parameters:
* I ->  ICASE  - Type of filter
*                1 - even number of coefficients, bandpass filter
*                2 - odd number of coefficients, bandpass filter
*                3 - even number of coefficients, differentiator or
*                    Hilbert transform
*                4 - odd number of coefficients, differentiator or
*                    Hilbert transform
* D ->  ALPHA  - Double precision coefficients of the canonical
*                response.  The number of coefficients M is uniquely
*                related to NTAP, the number of filter coefficients.
*                ICASE=1, M=(NTAP-1)/2+1
*                      2, M=NTAP/2
*                      3, M=(NTAP-1)/2-1
*                      4, M=NTAP/2
* D <-  H      - Array of resultant filter coefficients (double
*                precision)
* I ->  NTAP   - Number of filter coefficients
*
*
* Routines required:
*     None
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/25 05:17:27 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE IMPULS (ICASE, ALPHA, H, NTAP)


      INTEGER ICASE,NTAP
      INTEGER M,I

      DOUBLE PRECISION ALPHA(0:*),H(0:NTAP-1)


* General procedure (as far as one can generalize) for converting
* filter types to a canonical form:
*
* Let Z(n)=z**n .  The zero phase form of the filter response is
*
*          N-1
*   H(z) = SUM h(n) Z(-n+(N-1)/2) .
*          n=0
*
* This filter has M degrees of freedom due to the symmetries imposed
* on the coefficients.  We will express H(z) as
*
*   H(z) = P(z) C(z) ,
*
* where C(z) is symmetric, C(z)=C(1/z),
*
*           M-1
*   C(z) =  SUM  c(n) Z(-n) , with c(n)=c(-n) , and
*         n=-(M-1)
*
* P(z) is a term which is independent of the filter coefficients.  The
* structure of P(z) can be implied from the symmetries of H(z).
* Furthermore, express C(z) as
*
*   C(z) = [A(z) + A(1/z)]/2 ,
*
*           = 0.5 a(-n) ,  n=-(M-1),...,-1
*      c(n) = a(0) ,       n=0
*           = 0.5 a(n) ,   n=1,2,...,M-1 .
* Then
*               M-1
*   H(w) = P(w) SUM a(n) cos(wn) .
*               n=0

*----------
* Bandpass filter: odd N, symmetric coefficients h(n)=h(N-n-1)
*                  M=(N-1)/2+1
*
*                   M-1
*   H(z) = h(M-1) + SUM h(M-n-1) [Z(n) + Z(-n)]
*                   n=1
*
*        = B(1/z) + B(z) ,
*
*              M-1
* where B(z) = SUM b(n) Z(n) ,  with h(n) = b(M-n-1) , n=0,1,...,M-2 ,
*              n=0                        = 2 b(0) ,   n=M-1 .
*
* In this case, P(z)=1, A(z)=B(z),
*   h(n) = 0.5 a(M-n-1) , n=0,1,...,M-2
*        = a(0) ,         n=M-1 .

      IF (ICASE.EQ.1) THEN
        M=(NTAP-1)/2+1
        DO 120 I=0,M-2
          H(I)=0.5D0*ALPHA(M-I-1)
 120    CONTINUE
        H(M-1)=ALPHA(0)

        DO 140 I=M,NTAP-1
          H(I)=H(NTAP-I-1)
 140    CONTINUE

*----------
* Bandpass filter: even N, symmetric coefficients h(n)=h(N-n-1)
*                  M=N/2
*
*          M-1
*   H(z) = SUM h(M-n-1) [Z(n+1/2) + Z(-n-1/2)]
*          n=0
*
*        = Z(-1/2) [z B(1/z) + B(z)] ,
*
*              M-1
* where B(z) = SUM b(n) Z(n) , with h(n) = b(M-n-1) , n=0,1,...,M-1 .
*              n=0
* Let
*   Q(z) = z B(1/z) + B(z)
*        = P'(z) C(z) , where P(z) = Z(-1/2) P'(z) .
*
* Q(z) has coefficients q(n) for n=-M,...,M-1.
*
*   b(n) = q(n) , n=0,1,...,M-1
*
* From the definition of Q(z), it can be seen that z=-1 is a root.
*
*   P'(z) = z + 1 and P(z) = Z(1/2) + Z(-1/2) .
*
* With this choice, Q(z) = z C(z) + C(z) , with q(n) = c(n+1) + c(n) ,
* and C(z) = C(1/z).  Finally
*
*   H(z) = P(z) C(z) and H(w) = 2 cos(w/2) C(w) .
*
*        = 0.5 a(M-1) ,              n=0 ,
*   h(n) = 0.5 [a(M-n-1) + a(M-n)] , n=1,...,M-2 ,
*        = a(0) + 0.5 a(1) ,         n=M-1 .

      ELSE IF (ICASE.EQ.2) THEN
        M=NTAP/2
        IF (M.GT.1) THEN
          H(0)=0.5D0*ALPHA(M-1)
          DO 220 I=1,M-2
            H(I)=0.5D0*(ALPHA(M-I-1)+ALPHA(M-I))
 220      CONTINUE
          H(M-1)=ALPHA(0)+0.5*ALPHA(1)
        ELSE
          H(0)=ALPHA(0)
        END IF

        DO 240 I=M,NTAP-1
          H(I)=H(NTAP-I-1)
 240    CONTINUE

*----------
* Differentiator or Hilbert transform: odd N, anti-symmetric
* coefficients
*                   h(n)=-h(N-n-1), M=(N-1)/2
*
*          M-1
*   H(z) = SUM h(M-n-1) [Z(n) - Z(-n)],    h(M-1)=0 ,
*          n=1
*
*        = z B(1/z) - Z(-1) B(z) ,
*
*              M-1
* where B(z) = SUM b(n) Z(n) ,  with h(n) = b(M-n-1) , n=0,1,...,M-2 ,
*              n=0                        = 2 b(0) ,   n=M-1 .
* Let
*   Q(z) = Z(2) B(1/z) - B(z)
*        = P'(z) C(z) , where P(z) = Z(2) P'(z) .
*
* Q(z) has coefficients q(n) for n=-(M+1),...,M-1 (with q(-1)=0).
*
*   b(n) = -q(n) , n=0,1,...,M-1.
*
* From the definition of Q(z), it can be seen that z=1 and z=-1 are
* roots.
*
*  P'(z) = Z(2) - 1 , and P(z) = z - Z(-1) .
*
* With this choice, Q(z) = Z(2) C(z) - C(z) , with
* q(n) = c(n+2) + c(n) , and C(z) = C(1/z).  Finally
*
*  H(z) = P(z) C(z) and H(w) = 2j sin(w) C(w) .
*
*       = 0.5 a(M-1) ,              n=0 ,
*  h(n) = 0.5 a(M-2) ,              n=1
*       = 0.5 [a(M-n-1)-a(M-n+1)] , n=2,...,M-2 ,
*       = a(0) - 0.5 a(2) ,         n=M-1 .

      ELSE IF (ICASE.EQ.3) THEN
        M=(NTAP-1)/2
        IF (M.GE.3) THEN
          H(0)=0.5D0*ALPHA(M-1)
          H(1)=0.5D0*ALPHA(M-2)
          DO 320 I=2,M-2
            H(I)=0.5D0*(ALPHA(M-I-1)-ALPHA(M-I+1))
 320      CONTINUE
          H(M-1)=ALPHA(0)-0.5D0*ALPHA(2)
        ELSE IF (M.EQ.2) THEN
          H(0)=0.5D0*ALPHA(1)
          H(1)=ALPHA(0)
        ELSE IF (M.EQ.1) THEN
          H(0)=ALPHA(0)
        END IF

        H(M)=0.0
        DO 340 I=M+1,NTAP-1
          H(I)=-H(NTAP-I-1)
 340    CONTINUE

*----------
* Differentiator or Hilbert transform: even N, anti-symmetric
* coefficients
*                   h(n)=-h(N-n-1), M=N/2
*          M-1
*   H(z) = SUM h(M-n-1) [Z(n+1/2) - Z(-n-1/2)]
*          n=0
*
*        = Z(-1/2) [z B(1/z) - B(z)] ,
*
*              M-1
* where B(z) = SUM b(n) Z(n) , with h(n) = b(M-n-1) , n=0,1,...,M-1 .
*              n=0
* Let
*   Q(z) = z B(1/z) - B(z)
*        = P'(z) C(z) , where P(z) = Z(-1/2) P'(z) .
*
* Q(z) has coefficients q(n) for n=-M,...,M-1.
*
*   b(n) = -q(n) , n=0,1,...,M-1
*
* From the definition of Q(z), it can be seen that z=1 is a root.
*
*   P'(z) = z - 1 and P(z) = Z(1/2) - Z(-1/2) .
*
* With this choice, Q(z) = z C(z) - C(z) , with q(n) = c(n+1) - c(n) ,
* and C(z) = C(1/z).  Finally
*
*   H(z) = P(z) C(z) , H(w) = 2j sin(w/2) C(w) .
*
*        = 0.5 a(M-1) ,              n=0 ,
*   h(n) = 0.5 [a(M-n-1) - a(M-n)] , n=1,...,M-2 ,
*        = a(0) - 0.5 a(1) ,         n=M-1 .

      ELSE
        M=NTAP/2
        IF (M.GT.1) THEN
          H(0)=0.5D0*ALPHA(M-1)
          DO 420 I=1,M-2
            H(I)=0.5D0*(ALPHA(M-I-1)-ALPHA(M-I))
 420      CONTINUE
          H(M-1)=ALPHA(0)-0.5D0*ALPHA(1)
        ELSE
          H(0)=ALPHA(0)
        END IF

        DO 440 I=M,NTAP-1
          H(I)=-H(NTAP-I-1)
 440    CONTINUE

      END IF


      RETURN

      END
