*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE ERRMAX (LST, LFN, IS, LOPT, ERR, DEV, LIM,
*                        AD, X, Y, NEXT,
*                        NGRID, XGRID, DES, WT, DLIML, DLIMU)
*
*
* Purpose:
*     Find a local extremum (Remez exchange algorithm)
*
*
* Description:
*     This routine looks for a local maximum in the error curve.  The
*     routine checks a range of values on the approximating grid.  The
*     initial point and terminating point are specified.  This routine
*     continues as long as the error function is increasing.
*
*
* Parameters:
* I ->  LST    - Beginning position (1 to NGRID)
* I ->  LFN    - Search limit (1 to NGRID).  The search starts at LST
*                in the direction indicated by ID.  If LST is on the
*                wrong side of LFN (for instance LFN < LST when IS is
*                positive), no points are searched.
* I ->  IS     - Search mode.  The sign of IS indicates the search
*                direction, while the magnitude of IS indicates the
*                type of search
*                IS > 0 - upwards from LST
*                IS < 0 - downwards from LST
*                |IS|=1 - Continue searching as long as the error is
*                         is increasing.  If the first probe is not
*                         successful, the search is terminated.
*                |IS|=2 - Continue searching until a local maximum is
*                         found or the search range is exhausted
* I <-  LOPT   - Position of the extremum found (in the range LST to
*                LFN).  LST is set to -1 if no extremum with error
*                larger than the initial value of ERR is found.
* R <-> ERR    - This is the largest error found.  It is both an input
*                and an output variable.
* R ->  DEV    - Deviation at the design points
* R <-> LIM    - Type of extremum.  This is both an input and an output
*                value.  On input it indicates the type of extremum to
*                which ERR applies.  On output it indicates the type of
*                extremum found.
*                -2 - Lower constraint limit
*                -1 - Lower ripple
*                +1 - Upper ripple
*                +2 - Upper constraint limit
* D ->  AD     - Double precision array of Lagrange interpolation
*                coefficients
* D ->  X      - Double precision array of abscissa values for the
*                Lagrange interpolation
* D ->  Y      - Double precision array of ordinate values for the
*                Lagrange interpolation
* I ->  NEXT   - Number of interpolation points
* I ->  NGRID  - Number of elements in each of XGRID, DES, WT, DLIML,
*                and DLIMU
* R ->  XGRID  - Array of abscissa values (grid points).  These values
*                must be in decreasing order and lie in the range
*                [-1,+1].
* R ->  DES    - Desired values on the grid
* R ->  WT     - Weight values on the grid, WT(i) > 0
* R ->  DLIML  - Lower constraint values on the grid,
*                DLIML(i) <= DES(i)
* R ->  DLIMU  - Upper constraint values on the grid,
*                DLIMU(i) >= DES(i)
*
*
* Routines required:
*     VLGINT - Perform Lagrange interpolation
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/25 05:17:33 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE ERRMAX (LST, LFN, IS, LOPT, ERR, DEV, LIM,
     -                   AD, X, Y, NEXT,
     -                   NGRID, XGRID, DES, WT, DLIML, DLIMU)


      INTEGER LLIM,LDEV,UDEV,ULIM
      PARAMETER (LLIM=-2,LDEV=-1,UDEV=+1,ULIM=+2)

      INTEGER LST,LFN,IS,LOPT,LIM,NEXT,NGRID
      INTEGER ID,L,LL

      REAL ERR,DEV,XGRID(NGRID),DES(NGRID),WT(NGRID),
     -     DLIML(NGRID),DLIMU(NGRID)
      REAL RIP,CON,ERRT

      DOUBLE PRECISION AD(NEXT),X(NEXT),Y(NEXT)
      DOUBLE PRECISION VLGINT,PX


      LOPT=-1
      IF (IS.GE.0) THEN
        ID=+1
      ELSE
        ID=-1
      END IF
      DO 100 L=LST,LFN,ID

        PX=VLGINT(XGRID(L),AD,X,Y,NEXT)

        IF (LIM.EQ.UDEV .OR. LIM.EQ.ULIM) THEN

* Looking for an upward peak
          RIP=WT(L)*(PX-DES(L))
          CON=PX-DLIMU(L)+DEV
          IF (RIP.GE.CON) THEN
            LL=UDEV
            ERRT=RIP
          ELSE
            LL=ULIM
            ERRT=CON
          END IF

        ELSE

* Looking for a downward peak
          RIP=WT(L)*(DES(L)-PX)
          CON=DLIML(L)-PX+DEV
          IF (RIP.GE.CON) THEN
            LL=LDEV
            ERRT=RIP
          ELSE
            LL=LLIM
            ERRT=CON
          END IF
        END IF

* See if the error has increased
        IF (ERRT.GT.ERR) THEN
          LOPT=L
          LIM=LL
          ERR=ERRT
        ELSE IF (LOPT.GE.0 .OR. ABS(IS).EQ.1) THEN
          GO TO 200
        END IF

 100  CONTINUE


 200  RETURN

      END
