*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     PROGRAM DFIR
*
*
* Purpose:
*     Design FIR filters
*
*
* Description:
*     This program designs six types of finite impulse response
*     filters.
*     1. Equiripple Linear Phase Bandpass Filter (multiple passbands
*        and stopbands).  The filter is specified in terms of the
*        desired response in bands.  The band specifications include
*        desired value, relative weighting and constraint limits on the
*        allowed values in the band.  The resulting filter is a minmax
*        approximation (with constraints) to the given specifications.
*        The filter coefficients have an even symmetry about the middle
*        of the filter.
*     2. Equiripple Linear Phase Bandpass Filter (multiple passbands
*        and stopbands) with sin(x)/x compensation.  The filter is
*        specified in terms of the desired response in bands.  The band
*        specifications include desired value, relative weighting and
*        constraint limits on the allowed values in the band.  The
*        resulting filter is a minimax approximation (with constraints)
*        to the given specifications.  The filter coefficients have an
*        even symmetry about the middle of the filter.
*     3. Equiripple Differentiator.  The filter is specified in terms
*        of the desired slope in bands.  The band specifications
*        include desired slope, relative weighting and constraint
*        limits on the allowed slopes in the band.  The resulting
*        filter is a minimax approximation (with constraints) to the
*        given specifications.  The filter coefficients have an odd
*        symmetry about the middle of the filter.
*     4. Equiripple Hilbert Transform Filter. The filter is specified
*        in terms of the desired response in bands.  The band
*        specifications include desired value, relative weighting and
*        constraint limits on the allowed values in the band.  The
*        resulting filter is a minimax approximation (with constraints)
*        to the given specifications.  The filter coefficients have an
*        odd symmetry about the middle of the filter.
*     5. Interpolating Filter.  The design is specified in terms of the
*        power spectrum of the input signal.  For this given spectrum,
*        the mean-square interpolating error is minimized.  The filter
*        is linear phase, i.e., the coefficients are symmetrical (even)
*        about the middle of the filter.  If the number of coefficients
*        is odd, the filter coefficients have regularly spaced zero
*        crossings with a spacing equal to the interpolation ratio.
*     6. Equiripple Nyquist Filter.  The design is specified in terms
*        of a zero crossing interval and a stopband weighting.  The
*        resulting filter will be a minimax approximation to the
*        stopband specification with the constraint of the regular zero
*        crossings.  The filter is linear phase, with even symmetry.
*        The total number of coefficients is odd.
*     7. Minimum Phase Factor of an Equiripple Nyquist Filter.  The
*        design is specified in terms of a zero crossing interval and
*        a stopband weighting. The resulting filter will be a minimax
*        approximation to the stopband specification with the
*        constraint of the regular zero crossings.  The filter
*        generated by this program is the minimum phase factor of the
*        overall Nyquist filter. The other factor (maximum phase) can
*        be obtained by time reversing the coefficients of the minimum
*        phase factor.  The overall linear phase Nyquist filter is the
*        convolution of the minimum and maximum phase filters.
*
*     The most common cause of numerical difficulties in designing a
*     filter (signalled by error messages) is the use of too high an
*     order filter for the given specifications. This is particularly
*     so for Interpolating Filter and Nyquist Filter designs.  In all
*     cases, the quality of the final design should be checked by
*     plotting the frequency response curve.
*
*
* Parameters:
*     Data values can be specified in free format.  Multiple values can
*     be separated by spaces or commas.  Keywords can be in either
*     upper or lower case.
*
*       RECORD 1 - Filter_type
*          This is one of the keyword phrases given below (an asterisk
*          indicates the shortest acceptable abbreviation).
*          BPF - Multiple passband/stopband filter
*          REC*EIVING FILTER - Multiple passband/stopband filter used
*             as a receiving filter (sin(X)/X compensation)
*          INT*ERPOLATING FILTER - Interpolating filter
*          DIF*FERENTIATOR - Differentiator
*          HIL*BERT TRANSFORM FILTER - Hilbert transform filter
*          NYQ*UIST FILTER - Nyquist filter
*          MIN*IMUM PHASE - Minimum phase factor of a Nyquist filter
*
*     Bandpass filters, differentiators and Hilbert transform filters:
*
*       RECORD 2 - Ncof [Fsamp [Grid]]
*          This record contains 1, 2 or 3 items.  The last two are
*          optional.
*          1 - Number of filter coefficients (maximum 256)
*          2 - Sampling frequency.  If the sampling frequency is not
*              specified, the normalized value of one is used.  The
*              summary printout uses normalized frequency.
*          3 - Grid density.  The desired function value is translated
*              to a specification on a dense grid of frequency points.
*              The grid density is the number of grid points per
*              extremum of the response.  If the grid density is not
*              specified, it is set as large as possible (given the
*              array space available) up to a maximum of 16.  If the
*              keyword MAX*IMUM is specified, the maximum grid density
*              compatible with the array space is used.
*       RECORD 3 - [File_name]
*          This file is used to store the coefficient values.  If the
*          file name is all blank, the output file is not created;
*          instead the values are printed out.
*       Following RECORD(S) - Freq Value Weight [Low_V High_V]
*          The desired values, weights and constraint values are
*          specified in bands.  Within a band, tabulated values are
*          given.  Piecewise monotonic cubic interpolation is used
*          between the given tabulated points.  Bands are separated
*          with a transition band record.  The specifications must be
*          in order of increasing frequency.
*          - Transition band record (placed between bands): all blank
*            record or the keyword *TRANSITION BAND
*          - Tabulated band values:
*            5 items.  The last two values are constraints.  Within a
*            band, all or none of the points must have constraint
*            values.  The program approximates the best minimax filter
*            subject to the response lying between the given constraint
*            limits.
*            1 - Frequency
*            2 - Desired filter response value (or desired slope for
*                differentiators)
*            3 - Relative weighting for the value (or slope)
*            4 - Lower constraint limit on the filter response (or
*                slope for differentiators)
*            5 - Upper constraint limit on the filter response (or
*                slope for differentiators)
*            A maximum of 200 frequency points can be specified.  The
*            program expects specification records to be entered until
*            one of the following occurs.
*            i) A frequency equal to half of the sampling frequency
*               (normalized frequency of 0.5) is entered,
*            ii) two null lines have been entered (the first signifies
*                a transition band), or
*            iii) an end-of-file is encountered.
*
*     Interpolating filters:
*
*       RECORD 2 - Int_ratio
*          This record specifies the interpolation ratio.
*       RECORD 3 - Ncof [Fsamp]
*          This record has two values, the second being optional.
*          1 - Number of filter coefficients (maximum 399).  The filter
*              can be considered to be composed of IR subfilters, where
*              IR is the interpolating ratio.  A subfilter contains
*              every IR'th coefficient of the original filter.  The
*              maximum length of any subfilter is 42. This means that
*              there is an additional upperbound on the filter length
*              of 42*IR+1.  If the number of coefficients is odd, the
*              zero'th subfilter which includes the middle coefficient
*              will have coefficients 0, ... 0, 1, 0, ... 0, with the
*              unity coefficient being the middle coefficient of the
*              overall filter.  If the filter length is of the form
*              2*L*IR+1, the outer coefficients are both zero.
*          2 - Sampling frequency of the input sequence.  The sampling
*              frequency at the output of the interpolator is IR times
*              this value.  If the sampling frequency is not specified,
*              the normalized value of one is used.  The summary
*              printout uses a normalized sampling frequency.
*       RECORD 4 - [File_name]
*          This file is used to store the coefficient values.  If the
*          file name is all blank, the output file is not created;
*          instead the values are printed out.
*       Following RECORDS - Freq PSpecd [psd]    or   Freq Cpow cos
*          The remaining records give power spectrum values.  The power
*          spectrum of the input sequence is specified in terms of a
*          continuous component and sinusoidal components.  The
*          continuous component is given as tabulated values of the
*          power spectral density.  A piecewise monotonic cubic
*          interpolation is used between the given values.  The
*          sinusoidal components of the power spectrum are specified by
*          frequency and power.  Each input record specifies either a
*          power spectral density value or sinusoidal power value.  For
*          a power spectral density, the first field is the frequency,
*          the second field is the value of the power spectral density,
*          and the third field is either blank or contains the
*          identifying string "psd".  The continuous power spectrum
*          must be specified with frequency values which are in
*          increasing order.  The power spectral density is assumed to
*          be zero outside the given range of frequencies.  In
*          addition, the cubic interpolant for the continuous portion
*          of the power spectrum has zero slope at the first and last
*          points specified.  For a sinusoidal component, the first
*          field is the frequency, the second field the power of the
*          sinusoidal component, and the third field contains the
*          identifying string "cos".
*
*     Nyquist filters:
*
*       RECORD 2 - Nz
*          This record specifies the zero crossing interval (samples)
*       RECORD 3 - Ncof [Fsamp [Grid]]
*          This record contains 1, 2 or 3 items.  The last two tems are
*          optional.
*          1 - Number of filter coefficients for Nyquist filter (odd
*              number, maximum 255)
*          2 - Sampling frequency.  If the sampling frequency is not
*              specified, the normalized value of one is used. The
*              summary printout uses normalized frequency.
*       RECORD 4 - [File_name]
*          This file is used to store the coefficient values.  If the
*          file name is all blank, the output file is not created;
*          instead the values are printed out.
*       Following RECORD(S) - Freq Weight
*          The following records give the stopband weight
*          specifications for the Nyquist filter (2 items to a record).
*          Piecewise monotonic cubic interpolation is used between the
*          given tabulated weights. The values must be given in order
*          of increasing frequency.
*          1 - Frequency.  The first frequency specifies the lower edge
*              of the stopband (normally in the interval (Fs/(2*Nz) to
*              Fs/2, where Fs is the sampling frequency and Nz is the
*              zero crossing interval).
*          2 - Relative weighting for the frequency point.
*          A maximum of 200 frequency points can be specified.  The
*          program expects specification records to be entered until
*          one of the following occurs.
*          i) A frequency equal to half of the sampling frequency
*             (normalized frequency of 0.5) is entered,
*          ii) a null line is been entered, or
*          iii) an end-of-file is encountered.
*
*     Minimum Phase Factor of a Nyquist filter:
*
*       RECORD 2 - Nz
*          This record specifies the zero crossing interval (samples)
*       RECORD 3 - Ncof [Fsamp [Grid]]
*          This record contains 1, 2 or 3 items.  The last two tems are
*          optional.
*          1 - Number of filter coefficients for the minimum phase
*              factor (maximum 128)
*          2 - Sampling frequency.  If the sampling frequency is not
*              specified, the normalized value of one is used. The
*              summary printout uses normalized frequency.
*          3 - Grid density.  The desired stopband weights are
*              translated to a specification on a dense grid of
*              frequency points. The grid density is the number of grid
*              points per extremum of the stopband response. If the
*              grid density is not specified, it is set as large as
*              possible (given the array space available) up to a
*              maximum of 16.  If the keyword MAX*IMUM is specified,
*              the maximum grid density compatible with the array
*              space is used.
*       RECORD 4 - Coefficient file.  This files are used to store the
*          coefficient values of the minimum phase factor of the
*          Nyquist filter.  If a coefficient file is not specified, the
*          coefficients of the minimum minimum phase factor of the
*          Nyquist filter are printed out.
*       Following RECORD(S) - Freq Weight
*          The following records give the Stopband weight
*          specifications for the minimum phase factor (2 items to a
*          record).  Piecewise monotonic cubic interpolation is used
*          between the given tabulated weights.  The values must be
*          given in order of increasing frequency.
*          1 - Frequency.  The first frequency specifies the lower edge
*              of the stopband (normally in the interval (Fs/(2*Nz) to
*              Fs/2, where Fs is the sampling frequency and Nz is the
*              zero crossing interval).
*          2 - Relative weighting for the frequency point.
*          A maximum of 200 frequency points can be specified.  The
*          program expects specification records to be entered until
*          one of the following occurs.
*          i) A frequency equal to half of the sampling frequency
*             (normalized frequency of 0.5) is entered,
*          ii) a null line is been entered, or
*          iii) an end-of-file is encountered.
*
*     Examples -
*     32 tap constrained bandpass filter:
*     Stopbands are from 0 to 0.1 and 0.425 to 0.5, and the passband is
*     from 0.2 to 0.35.  The relative weights are 10 in the stopbands
*     and 1 in the passband. The response is constrained to be positive
*     (between 0 and 1) in the first stopband. The filter coefficients
*     will be written to file filt.cof.
*         bpf
*         32
*         filt.cof
*         0   0 10 0 1
*         0.1 0 10 0 1
*           Transition Band
*         0.2  1 1
*         0.35 1 1
*           Transition Band
*         0.425 0 10
*         0.5   0 10
*     23 tap interpolating filter:
*     The input spectrum is modelled by a power spectrum with a
*     transition band from 0.3 to 0.4 and a dc component.  The
*     interpolating ratio is 4.  The coefficients are written to file
*     fili.cof.
*         Int
*         4
*         23
*         fili.cof
*         0   1000  cos
*         0   1
*         0.3 1
*         0.4 0
*     15 tap minimum phase factor of a Nyquist filter:
*     The zero crossing interval is 4.  An excess bandwidth of 50% is
*     used.  The coefficients of the minimum phase factor are written
*     to file filn.cof.  An equiripple design is used.
*         MIN
*         4
*         15
*         filn.cof
*         0.1875, 1
*         0.5,    1
*
*
* Routines required:
*     FIRDSG - Design a linear phase FIR filter
*     HALT   - Print an error message, stop with error status set
*     INTDSG - Design a min. mean-square error interpolating filter
*     KEYUPC - Match keyword strings (case insensitive)
*     LENSTR - Find the length of a whitespace terminated string
*     MPHDSG - Design a factored Nyquist filter
*     NYQDSG - Design a Nyquist filter
*     PRTCOF - Print FIR filter coefficients
*     PRTFIR - Print FIR filter design parameters
*     PRTINT - Print interpolating filter design parameters
*     PRTMPH - Print Nyquist filter parameters (factored filter)
*     PRTNYQ - Print Nyquist filter design parameters
*     RDLINE - Read a line of input from standard input
*     REDFIR - Read FIR filter specifications
*     REDINT - Read interpolating filter specifications
*     REDMPH - Read Nyquist filter specifications (factored filter)
*     REDNYQ - Read Nyquist filter specifications
*     WRTFIR - Write an FIR filter coefficient file
*     WRTINT - Write an interpolating filter coefficient file
*     WRTMPH - Write a Nyquist filter file (factored filter)
*     WRTNYQ - Write a Nyquist filter coefficient file
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.4 $  $Date: 1993/01/25 20:16:45 $
*
*
*-----------------------------------------------------------------------

      PROGRAM DFIR


      INTEGER NKEY,BPF,REC,DIF,HIL,INT,NYQ,MPH
      INTEGER MXNCOF,MXPT
      PARAMETER (NKEY=7,BPF=1,REC=2,DIF=3,HIL=4,INT=5,NYQ=6,MPH=7)
      PARAMETER (MXNCOF=399,MXPT=200)

      CHARACTER FNAME*255,LINE*80
      CHARACTER KEYTAB(NKEY)*25

      INTEGER NCHR,IEOF,ITYPE,NTAP,IER
      INTEGER IR,NVAL(2)
      INTEGER NN,NF
      INTEGER NBANDS,IBAND(MXPT)
      INTEGER LENSTR,KEYUPC

      REAL ERR2
      REAL GRIDD,DEVS
      REAL H(0:MXNCOF-1),FREQ(MXPT),VAL(MXPT),WEIGHT(MXPT),
     -     VLIML(MXPT),VLIMU(MXPT)

      DATA KEYTAB/'BPF',
     -            'REC*EIVING FILTER',
     -            'DIF*FERENTIATOR',
     -            'HIL*BERT TRANSFORM FILTER',
     -            'INT*ERPOLATING FILTER',
     -            'NYQ*UIST FILTER',
     -            'MIN*IMUM PHASE'/


* Read the filter type
      CALL RDLINE('Filter type: ',LINE,NCHR,IEOF)
        IF (NCHR.EQ.0) GO TO 900
      ITYPE=KEYUPC(LINE,KEYTAB,NKEY)
      IF (ITYPE.EQ.0) CALL HALT('DFIR - Invalid filter type')

* Interpolating filter design
      IF (ITYPE.EQ.INT) THEN
        CALL REDINT(FNAME,NTAP,IR,FREQ,VAL,NVAL)
        CALL INTDSG(H,NTAP,IR,FREQ,VAL,NVAL,ERR2,IER)
        IF (IER.NE.0) CALL HALT('DFIR - Error detected in INTDSG')

* Nyquist filter design
      ELSE IF (ITYPE.EQ.NYQ) THEN
        CALL REDNYQ(FNAME,NTAP,NN,NF,FREQ,WEIGHT,GRIDD)
        CALL NYQDSG(H,NTAP,NN,NF,FREQ,WEIGHT,GRIDD,DEVS,IER)
        IF (IER.NE.0) CALL HALT('DFIR - Error detected in NYQDSG')

* Minimum phase factor of a Nyquist filter
      ELSE IF (ITYPE.EQ.MPH) THEN
        CALL REDMPH(FNAME,NTAP,NN,NF,FREQ,WEIGHT,GRIDD)
        CALL MPHDSG(H,NTAP,NN,NF,FREQ,WEIGHT,GRIDD,DEVS,IER)
        IF (IER.NE.0) CALL HALT('DFIR - Error detected in MPHDSG')

* Bandpass, differentiator or Hilbert transform design
      ELSE
        CALL REDFIR(ITYPE,NTAP,FNAME,GRIDD,IBAND,NBANDS,
     -              FREQ,VAL,WEIGHT,VLIML,VLIMU)
        CALL FIRDSG(H,NTAP,ITYPE,GRIDD,IBAND,NBANDS,
     -              FREQ,VAL,WEIGHT,VLIML,VLIMU,DEVS,IER)
        IF (IER.NE.0) CALL HALT('DFIR - Error detected in FIRDSG')

      END IF

* Summary printout
      IF (LENSTR(FNAME).EQ.0) THEN
        CALL PRTCOF(ITYPE,H,NTAP)
      END IF

      IF (ITYPE.EQ.INT) THEN
        CALL PRTINT(IR,ERR2,FREQ,VAL,NVAL)
        CALL WRTINT(FNAME,H,NTAP,IR,FREQ,VAL,NVAL)

      ELSE IF (ITYPE.EQ.NYQ) THEN
        CALL PRTNYQ(DEVS,NN,NF,FREQ,WEIGHT)
        CALL WRTNYQ(FNAME,H,NTAP,NN,NF,FREQ,WEIGHT)

      ELSE IF (ITYPE.EQ.MPH) THEN
        CALL PRTMPH(DEVS,NN,NF,FREQ,WEIGHT)
        CALL WRTMPH(FNAME,H,NTAP,NN,NF,FREQ,WEIGHT)

      ELSE
        CALL PRTFIR(ITYPE,DEVS,IBAND,NBANDS,
     -              FREQ,VAL,WEIGHT,VLIML,VLIMU)
        CALL WRTFIR(FNAME,ITYPE,H,NTAP,
     -              IBAND,NBANDS,FREQ,VAL,WEIGHT,VLIML,VLIMU)

      END IF


 900  END
