*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     DOUBLE PRECISION FUNCTION CCPINT (X1, X2, Y1, Y2, D1, D2)
*
*
* Purpose:
*     Evaluate a cubic times cosine integral
*
*
* Description:
*     This function calculates the value of an integral of a cubic
*     function times a cosine.  The integral is
*
*            x2
*       P = Int (a x**3 + b x**2 + c x + d) cos(x) dx .
*            x1
*
*     The polynomial coefficients a, b, c, and d are determined from
*     the values and derivatives at the end points.
*
*
* Parameters:
* D <-  CCPINT - Output value (double precision)
*
* D ->  X1     - Abscissa value at the first point (double precision)
* D ->  X2     - Abscissa value at the second point (double precision)
* D ->  Y1     - Value at the first point (double precision)
* D ->  Y2     - Value at the second point (double precision)
* D ->  D1     - Derivative at the first point (double precision)
* D ->  D2     - Derivative at the second point (double precision)
*
*
* Routines required:
*     None
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/25 05:05:06 $
*
*
*-----------------------------------------------------------------------

      DOUBLE PRECISION FUNCTION CCPINT (X1, X2, Y1, Y2, D1, D2)


      DOUBLE PRECISION X1,X2,Y1,Y2,D1,D2
      DOUBLE PRECISION DX,S,DD1,DD2
      DOUBLE PRECISION A,B,C,D,AP,BP,CP,DP
      DOUBLE PRECISION FI,X

* Statement function to calculate the indefinite integral
      FI(X) = ( (C-6.*A) + X*(2.*B + X*3.*A) ) * COS(X)
     -        + ( (D-2.*B) + X*((C-6.*A) + X*(B + X*A)) ) * SIN(X)

* Express the cubic as
*
*   R(u) = a'' u**3 + b'' u**2 + c'' u + d'' ,
*
* where u=(x-x1)/(x2-x1) varies between 0 and 1 as x varies between
* x1 and x2.  The coefficients of P(u) can be determined by solving
* the simultaneous equations:
*
*    R(0) = y1       | 1  0  0  0 |  | d''|   | y1 |
*    R(1) = y2,  or  | 1  1  1  1 |  | c''| = | y2 |
*   R'(0) = d1'      | 0  1  0  0 |  | b''|   | d1'|   d1'=d1 (x2-x1)
*   R'(1) = d2'      | 0  1  2  3 |  | a''|   | d2'|   d2'=d2 (x2-x1)
*
* The polynomial coefficients for P(u) are:
*   d'' = y1                     = y1
*   c'' = d1'                    = d1 dX
*   b'' = 3(y2-y1) - 2 d1' - d2' = (3S - 2d1 - d2) dX
*   a'' = -2(y2-y1) + d1' + d2'  = (-2S + d1 + d2) dX
* where S=(y2-y1)/(x2-x1) and dX=(x2-x1)
*
* Define
*   Q(v)=a' v**3 + b' v**2 + c' v + d' ,
* where v = x-x1 = u dX.  The coefficients of Q(v) can be related to
* those of R(u),
*
*   d' = d''        = y1
*   c' = c''/dX     = d1
*   b' = b''/dX**2  = (3S - 2 d1 - d2)/dX
*   a' = a''/dX**3  = (-2S + d1 + d2)/dX**2

      IF (X1.NE.X2) THEN

        DX=X2-X1
        S=(Y2-Y1)/DX
        DD1=(D1-S)/DX
        DD2=(D2-S)/DX
        DP=Y1
        CP=D1
        BP=-(DD1+DD2+DD1)
        AP=(DD1+DD2)/DX

* Finally, consider
*   P(x) = a x**3 + b x**2 + c x + d ,
* where x=v+x1.  Substituting v=x-x1 in Q(v) and expanding out the
* terms, we get
*
*  d = -a' x1**3 + b' x1**2 - c' x1 + d'
*  c = 3 a' x1**2 - 2 b' x1 + c'
*  b = -3 a' x1 + b'
*  a = a'

        D=DP - X1*(CP - X1*(BP - X1*AP))
        C=CP - X1*(2.*BP - 3.*X1*AP)
        B=BP - 3.*X1*AP
        A=AP

* Indefinite integral formulas:
*       Int cos(x) dx = sin(x)
*     Int x cos(x) dx = cos(x) + x sin(x)
*     Int x**2 cos(x) = 2x cos(x) + (x**2-2) sin(x)
*  Int x**3 cos(x) dx = 3(x**2-2) cos(x) + (x**3-6x) sin(x)
*
* Combining these, the integral is
*
*  Int (a x**3 + b x**2 + c x + d) cos(x) dx
*               = [c + 2xb + 3a(x**2-2)] cos(x)
*                 + [d + cx + b(x**2-2) + a(x**3-6x) ] sin(x)

* Evaluate the integral
        CCPINT=FI(X2)-FI(X1)

      ELSE

* Zero width interval
        CCPINT=0.0

      END IF


      RETURN

      END
