/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  double INevCorr (double t, const struct PSD *psd, const struct PSD *dis)

Purpose:
  Evaluate a correlation given power spectrum values

Description:
  This function returns the value of an autocorrelation function at a given
  time lag.  The correlation function is calculated as the Fourier transform
  of a power spectrum.  The corresponding correlation is real and symmetrical
  about time lag zero.

  The power spectrum is specified in tabular form.  A piecewise monotonic cubic
  interpolation is used between the given power spectrum values.  The power
  spectrum is specified for non-negative frequencies.  Additional discrete
  (sinusoidal) components can be specified for the power spectrum.

Parameters:
  <-  double INevCorr
      Returned autocorrelation value
   -> double t
      Time lag value at which the correlation is to be evaluated
   -> const struct PSD *psd
      Structure describing the continuous power spectral density.  This
      structure contains an array of normalized frequencies and an array of
      the corresponding power spectral values.
   -> const struct PSD *dis
      Structure describing the discrete (sinusoidal) components of the power
      spectrum.  This structure contains an array of normalized frequencies and
      an array of the corresponding powers of the sinusoids.

Author / revision:
  P. Kabal  Copyright (C) 1995
  $Revision: 1.5 $  $Date: 1995/05/26 00:25:16 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: INevCorr.c 1.5 1995/05/26 FilterDesign-V1R7a $";

#include <math.h>
#include <libtsp.h>
#include "DFiltInt.h"

#define PIx2		6.28318530717958647693

double INevCorr (t, psd, dis)

     double t;
     const struct PSD *psd;
     const struct PSD *dis;

{
  int i;
  double corr;

/* Evaluate the correlation function */

  if (psd->N > 0)
    corr = INtimeResp (t, psd->f, psd->psd, psd->N);
  else
    corr = 0.0;

  for (i = 0; i < dis->N; ++i) {
    if (dis->f[i] < 0.0)
      UThalt ("INevCorr: Negative frequency");
    if (dis->psd[i] < 0.0)
      UThalt ("INevCorr: Negative sinusoid amplitude");
    corr = corr + dis->psd[i] * cos (PIx2 * dis->f[i] * t);
  }

  /* Return the function value */
  return corr;
}
