/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  void INdesInt (float h[], int Ncof, double Delay, int Ir,
                 const struct PSD *psd, const struct PSD *dis)

Purpose:
  Design a minimum mean-square error interpolating filter

Description:
  This procedure calculates the coefficients for an interpolating filter which
  minimizes the mean-square interpolation error given the power spectrum of
  the input signal.  The power spectrum of the input signal is specified in
  terms of a continuous component and discrete (sinusoidal) components.  The
  continuous component is given as a tabulated function.  A piecewise monotonic
  cubic interpolation is used between the given values.  The sinusoidal
  components are specified by frequency and power.

  This routine implements the method outlined by Oetken, Parks, and Schussler
  for a general power spectrum specification.
  Reference: G. Oetken, T. W. Parks and H. W. Schussler,
             "New results in the design of digital interpolators",
             IEEE Trans. ASSP, pp. 301-309, June 1975.

Parameters:
  <-  float h[]
      Resultant vector of filter coefficients
   -> int Ncof
      Number of filter coefficients
   -> double Delay
      Filter delay (samples)
   -> int Ir
      Interpolation ratio
   -> const struct PSD *psd
      Structure describing the continuous power spectral density.  This
      structure contains an array of normalized frequencies and an array of
      the corresponding power spectral values.
   -> const struct PSD *dis
      Structure describing the discrete (sinusoidal) components of the power
      spectrum.  This structure contains an array of normalized frequencies and
      an array of the corresponding powers of the sinusoids.

Author / revision:
  P. Kabal  Copyright (C) 1996
  $Revision: 1.10 $  $Date: 1996/06/17 21:27:40 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: INdesInt.c 1.10 1996/06/17 FilterDesign-V1R7a $";

#include <math.h>
#include <libtsp.h>
#include "DFiltInt.h"

#define ICEILV(n, m)	(((n) + ((m) - 1)) / (m))	/* int n,m >= 0 */

void
INdesInt (h, Ncof, Delay, Ir, psd, dis)

     float h[];
     int Ncof;
     double Delay;
     int Ir;
     const struct PSD *psd;
     const struct PSD *dis;

{
  double *rxx, *p, *hs;
  double fd0, del;
  int Nm, Nmmax;
  int i, k, m, n;
  int i0, klim;
  int ier;

/*
   We pick a point in the filter to serve as a reference point.  This point
   is Delay samples (measured in samples at the higher sampling rate) from the
   beginning of the filter.  Using this reference point, each coefficient can
   be associated with a time value.
     coefficient   relative time
       h[n]        n-Delay

   The interpolating filter is divided into Ir subfilters.  Consider the m'th
   coefficient of the i'th subfilter, 0 <= i < Ir, m = 0, 1, ... .  This
   indexing scheme assigns i = 0 to the subfilter which contains the first
   coefficient of the overall filter.
     coefficient        relative time
     hs[i](m)     <-->  m*Ir+i-Delay

   The equations to be solved for the i'th subfilter are
     R hs[i] = p[i] ,
   where the matrix R is Toeplitz with elements R(m,k) = rxx(m-k)
         the vector hs[i] has elements hs[i](m),
         the vector p[i] has elements p[i](m) = rxx((m+(i-Delay)/Ir), and
         the function rxx(t) gives the correlation at lag t*Ir.

   The time delay from the beginning of subfilter i to the reference point is
     D[i] = Delay - i.
   Consider another indexing scheme for the subfilters.  This index measures
   the "fractional" delay associated with the delay D[i],
     fd[i] = Ir*Ceil(D[i]/Ir) - D[i].
   Furthermore we can write the fractional delay as
     fd[i] = k[i] + a,
   where k[i] = Floor(fd[i]) and 0 <= a < 1.

   To take advantage of filter symmetries, it will be convenient to evaluate
   the subfilters in order of increasing fractional delay, or equivalently in
   order of increasing k[i].  Note that k[i] is an integer, 0 <= k[i] < Ir.
   Furthermore, adding one to i, increases k[i] by one, modulo Ir.  Thus k[i]
   can be expressed as,
     k[i] = (k[0] + i) mod Ir.
   The value k[0] is determined by substituting i=0, in the equations above,
     k[0] = Floor(Ir*Ceil(Delay/Ir) - Delay).
   Then for a given value for k, we can find i from
     i = (k - k[0]) mod Ir.  [Note: the % operator can return a negative value]

   The interpolating filter is symmetrical for Delay = (Ncof-1)/2,
     h[n] = h[Ncof-n-1].
   The symmetry of the overall filter imposes a time-reverse symmetry on the
   subfilters.
   Ncof odd:
     Subfilter i with k[i] = 0, has elements [0,...0,1,0,...,0].  Subfilter i
     with 1 <= k[i] < Ir/2 is the time-reversal of subfilter j with
     k[j] = Ir-k[i].
   Ncof even:
     Subfilter i with 0 <= k[i] < Ir/2 is the time-reversal of subfilter j
     with k[j] = Ir-k[i]-1.
   In either case only subfilters with k[i] = 0, ... , Ir/2 need to be
   explicitly calculated.  The remaining filters can be determined from
   the symmetries.
*/

/* Allocate memory for the arrays */
  Nmmax = ICEILV (Ncof, Ir);
  rxx = (double *) UTmalloc (Nmmax * sizeof (double));
  p = (double *) UTmalloc (Nmmax * sizeof (double));
  hs = (double *) UTmalloc (Nmmax * sizeof (double));

/* Fill in the autocorrelation vector */
  for (m = 0; m < Nmmax; ++m)
    rxx[m] = INevCorr ((double) m, psd, dis);

/* Find the mapping between k and i */
  fd0 = (Ir * ceil (Delay / Ir) - Delay);
  i0 = (Ir - (int) fd0) % Ir;

/* Check for symmetry */
  if (Delay == 0.5 * (Ncof-1))
    klim = Ir/2;
  else
    klim = Ir-1;

/* Loop over subfilter indices */
  for (k = 0; k <= klim; ++k) {

    i = (i0 + k) % Ir;
    Nm = MSiCeil (Ncof-i, Ir);
    if (Nm > Nmmax)
      UThalt ("INdesInt: Consistency check failed");

    /* Special case: delays (m*Ir+i-Delay) are a multiple of Ir */
    if (k == 0 && fd0 == (int) (fd0) &&
	Delay >= 0.0 && Delay <= Ncof-1) {
      for (m = 0; m < Nm; ++m) {
	if (m*Ir+i == Delay)
	  hs[m] = 1.0;
	else
	  hs[m] = 0.0;
      }
    }

    else {

      /* Fill in the right-hand side vector, p[i](.) */
      for (m = 0; m < Nm; ++m) {
	del = (-Delay + m*Ir + i) / Ir;
	p[m] = INevCorr (del, psd, dis);
      }

      /* Find the coefficients of the subfilter, solving A hs[i] = p[i] */
      ier = INdTpSolve (rxx, p, hs, Nm);
      if (ier != 0)
	UThalt ("INdesInt: Non-positive definite equations");
    }

    /* Copy the subfilter coefficients into the final filter */
    for (m = 0; m < Nm; ++m)
      h[m*Ir+i] = hs[m];

  }

/* For symmetric filters, fill in the rest of the filter from symmetry */
  for (k = klim+1; k < Ir; ++k) {
    i = (i0 + k) % Ir;
    Nm = MSiCeil (Ncof-i, Ir);
    for (m = 0; m < Nm; ++m) {
      n = m*Ir + i;
      h[n] = h[Ncof-n-1];
    }
  }


/* Deallocate the temporary arrays */
  UTfree (rxx);
  UTfree (p);
  UTfree (hs);

}
