/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  int INdTpSolve (const double R[], const double [], double c[], int N)

Purpose:
  Solve a Toeplitz set of equations

Description:
  This subroutine solves the set of matrix equations

    R c = g

  where R is an N by N symmetric Toeplitz matrix and c and r are N element
  column vectors.  The Toeplitz matrix R has elements which are equal along the
  diagonals, i.e. R(i,j)=r(abs(i-j)).

  This subroutine uses Levinson's method to calculate the solution.  In the
  case that either numerical instability or an inappropriate set of matrix
  coefficients results in a matrix which is not positive definite or singular,
  an error code is returned.

  This routine uses:
      2N-1    divides, and
    2N^2-3N   multiplies and adds.

  Reference:
    E. A. Robinson, "Multichannel time series analysis with digital computer
    programs", Holden-Day, 1967, p. 44.

Parameters:
  <-  int INdTpSolve
      Error flag, 0 - no error
                  1 - matrix not positive definite
   -> const double R[]
      Vector specifying the first column of the Toeplitz matrix
   -> const double g[]
      Righthand side vector
  <-  double d[]
      Solution vector
   -> int N
      Number of equations

Author / revision:
  P. Kabal  Copyright (C) 1995
  $Revision: 1.5 $  $Date: 1995/02/01 21:21:29 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: INdTpSolve.c 1.5 1995/02/01 FilterDesign-V1R7a $";

#include <libtsp.h>
#include "DFiltInt.h"

int
INdTpSolve (R, g, c, N)

     const double R[];
     const double g[];
     double c[];
     int N;

{
  int i, k, l;
  double perr, rc, t, sum;
  double *pc;

  
/* Allocate a temporary array */
  pc = (double *) UTmalloc (N * sizeof (double));

  perr = R[0];
  if (perr <= 0.0)
    return 1;

  c[0] = g[0] / perr;
  for (k = 0; k < N - 1; ++k) {

    sum = R[k+1];
    for (i = 0, l = k; i < k; ++i, --l)
      sum = sum - R[l] * pc[i];
    rc = -sum / perr;

/*
   Calculate the prediction error (equivalent to perr = perr * (1-rc^2))
   A change in sign of perr means that rc has a magnitude greater than unity
   (corresponding to a non-positive definite system of equations)
*/
    perr = perr + rc * sum;
    if (perr <= 0.0)
      return 1;

    pc[k] = -rc;
    for (i = 0, l = k - 1; i < l; ++i, --l) {
      t = pc[i] + rc * pc[l];
      pc[l] = pc[l] + rc * pc[i];
      pc[i] = t;
    }
    if (i == l)
      pc[i] = pc[i] + rc * pc[i];

    sum = g[k+1];
    for (i = 0, l = k + 1; i <= k; ++i, --l)
      sum = sum - c[i] * R[l];
    c[k+1] = sum / perr;

    for (i = 0, l = k; i <= k; ++i, --l)
      c[i] = c[i] - c[k+1] * pc[l];

  }
/* Deallocate the temporary array */
  UTfree (pc);

  return 0;
}
