/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  DFiltInt [options] [FiltFile]

Purpose:
  Design a minimum mean-square error FIR interpolating filter

Description:
  This routine designs an interpolating filter which minimizes the mean-square
  interpolation error.  The input to the interpolation process is a basic rate
  sampled signal.  The output of the interpolation process is an increased rate
  sampled signal.  The design of the interpolating filter is specified in terms
  of the power spectrum of the underlying continuous-time signal.  This power
  spectrum is used to derive the auto-correlation functions for the sampled
  signals.  The auto-correlation determines the mean-square error between the
  increased-rate signal and the samples of the underlying continuous-time
  signal.

  Conceptually, interpolation has two steps.  In the first, the sampling rate
  of the input sequence is increased by a factor of Ir, by inserting Ir-1 zeros
  between each input sample.  In the second step, the increased rate sequence
  is applied to the interpolating filter to form the interpolated sequence.

  The filter delay can be specified, but defaults to (Ncof-1)/2, i.e. the
  middle of a filter with Ncof coefficients.  If the delay corresponds to the
  middle of the filter, the filter will be linear phase, i.e., the coefficients
  will be symmetrical about the middle of the filter.

  This program implements the method outlined by Oetken, Parks, and Schussler
  for a general power spectrum specification.
  Reference:
      G. Oetken, T. W. Parks and H. W. Schussler, "New results in the design of
      digital interpolators", IEEE Trans. ASSP, pp. 301-309, June 1975.

Parameters:
  This program reads options from the command line and from standard input.
  At a minimum the command line must specify the interpolating ratio and the
  number of coefficients.  If the filter file name is not specified, the filter
  coefficients are printed out.
  -i IR, --interpolate=IR
      Interpolation ratio.
  -n NCOF, --number_coefficients=NCOF
      Number of filter coefficients.
  -d DELAY, --delay=DELAY
      Filter delay.  The default value is (NCOF-1)/2.
  -s SFREQ, --srate=SFREQ
      Sampling frequency for the sequence to be interpolated.  The sampling
      frequency for the interpolated sequence is this value multiplied by the
      interpolating ratio.  If the sampling frequency is not specified, a
      normalized sampling frequency of one is used.
  -h, --help
      Print a list of options and exit.
  -v, --version
      Print the version number and exit.

  The input power spectrum specifications are read from standard input.
   Frequency, Value:  freq value [psd | cos]
  These records specify the power spectrum values.  The one-sided power
  spectrum of the input signal is specified in terms of a continuous component
  and discrete (sinusoidal) components.  The continuous component is given as
  tabulated values of the power spectral density.  Piecewise monotonic cubic
  interpolation is used between the given values.  The sinusoidal components of
  the power spectrum are specified by frequency and power.  Each input record
  specifies either a power spectral density value or sinusoidal power value.
  For a power spectral density, the first field is the frequency (positive
  values), the second field is the value of the power spectral density, and the
  third field is either blank or contains the identifying string "psd".  The
  continuous power spectrum must be specified with positive frequency values
  which are in increasing order.  The power spectral density is assumed to be
  zero outside the given range of frequencies.  For a sinusoidal component, the
  first field is the frequency, the second field the power of the sinusoidal
  component, and the third field contains the identifying string "cos".

  Example, 23 tap linear phase interpolating filter:
      The input spectrum is modelled by a power spectrum which is flat from dc
      to 2400 Hz.  The spectrum then decreases to become zero at 3200 Hz.
      There is an additional dc component.  The interpolating ratio is 4
      (input 8000 Hz, output 32000 Hz).  The coefficients are written to file
      filtx4.cof.
       % DFiltInt -i 4 -n 23 -s 8000 -f filtx4.cof
       Input signal power spectrum:
         Frequency, Value: 0 1000  cos
         Frequency, Value: 0   1
         Frequency, Value: 2400 1
         Frequency, Value: 3200 0

Author / version:
  P. Kabal / v2r1  1996/06/16  Copyright (C) 1996

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: DFiltInt.c 1.21 1996/06/17 FilterDesign-V1R7a $";

#include <stdlib.h>		/* prototype for exit */
#include <libtsp.h>
#include "DFiltInt.h"

#ifndef EXIT_SUCCESS
#  define EXIT_SUCCESS	0	/* Normally in stdlib.h */
#endif

int
main (argc, argv)

     int argc;
     const char *argv[];

{
  int Ir, Ncof;
  float *h;
  const char *Fname;
  float Delay, Sfreq;
  float *mse;
  struct PSD psd;
  struct PSD dis;
  
/* Read the interpolation parameters */
  INoptions (argc, argv, &Ncof, &Delay, &Ir, &Sfreq, &Fname);
  INreadSpec ((double) Sfreq, &psd, &dis);

/* Design the filter */
  h = (float *) UTmalloc (Ncof * sizeof (float));
  INdesInt (h, Ncof, (double) Delay, Ir, &psd, &dis);

/* Calculate the mean-square error */
  mse = INintMSE (h, Ncof, (double) Delay, Ir, &psd, &dis);
   
/* Print summary and write coefficients to a file */
  if (Fname == NULL)
    VRfPrint (stdout, "Interpolating Filter:", h, Ncof);
  INprintSpec (Ncof, (double) Delay, Ir, mse, (double) Sfreq, &psd, &dis);
  if (Fname != NULL)
    INwriteCof (Fname, h, Ncof, (double) Delay, Ir, &psd, &dis);

  UTfree (h);

  return EXIT_SUCCESS;
}
