/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  void MMtrCheb (const double acof[], int Ncof, double c, double d,
                 double bcof[])

Purpose:
  Transform Chebyshev polynomial coefficients

Description:
  The inputs to this routine are a set of coefficients for an expansion in
  Chebyshev polynomials in the variable x.  This routine finds the
  corresponding set of coefficients for an expansion in Chebyshev polynomials
  in the variable y, where y=cx+d.

  The expansions in Chebyshev polynomials are

    N-1               N-1
    SUM a(n) T(n,x) = SUM b(n) T(n,cx+d) ,  where
    n=0               n=0

  T(n+1,x) = 2x T(n,x) - T(n-1,x)  with initial conditions T(0,x)=1 and
  T(1,x)=x.

Parameters:
   -> const double acof[]
      Array of input coefficients
   -> int Ncof
      Number of elements in acof and bcof
   -> double c
      Transformation scaling value.  This value must be nonzero.
   -> double d
      Transformation offset value
  <-  double bcof[]
      Array of transformed coefficients

Author / revision:
  P. Kabal  Copyright (C) 1995
  $Revision: 1.4 $  $Date: 1995/02/09 22:12:55 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: MMtrCheb.c 1.4 1995/02/09 FilterDesign-V1R7a $";

#include <libtsp.h>
#include "DFiltFIR.h"

void
MMtrCheb (acof, Ncof, c, d, bcof)

     const double acof[];
     int Ncof;
     double c;
     double d;
     double bcof[];

{
  double *bnp;
  double *bnpp;
  int i, n, k;
  double p, q;
  double bb;

/*
   Use a shortcut for c=1 and d=0.  The general formulation can be used, but
   with a great deal of extra data shuffling.
*/
  if (c == 1.0 && d == 0.0) {
    for (i = 0; i < Ncof; ++i)
      bcof[i] = acof[i];
  }

  else {

/* General case */

/* Allocate array space */
    bnpp = (double *) UTmalloc ((Ncof - 2) * sizeof (double));
    bnp = (double *) UTmalloc ((Ncof - 1) * sizeof (double));

/* Define the partial sum with k terms as

             N-1
    S(k,x) = SUM a(n) T(n-N+k,x).
            n=N-k

  The partial sum can be calculated recursively,

    S(k,x) = 2x S(k-1,x) - S(k-2,x) - x a(N-k+1) + a(N-k),
             k=1,...,N  with initial conditions S(0,x) = S(-1,x) = 0.

  This difference equation describes a linear system with inputs x a(N-k+1)
  and a(N-k).  Consider the simpler difference equation

    S'(k,x) = 2x S'(k-1,x) - S'(k-2,x) + a(N-k), k=1,...,N              (1)
                               with initial conditions S'(0,x) = S'(-1,x) = 0.

  Then S'(k,x) is the output for the input a(N-k) and x S'(k-1,x) is the output
  for the input x a(N-k+1).  The overall partial sum is

    S(k,x) = S'(k,x) - x S'(k-1,x).                                     (2)

  Consider a related polynomial expansion in y=cx+d,

              k-1
    S'(k,x) = SUM b'(n,k) T(n,y).
              n=0

  Using (1), the expansion for S'(k,x) can be manipulated to become

    k-1
    SUM [ b'(n,k) - 2x b'(n,k-1) + b'(n,k-2) ] T(n,y) - a(N-k) = 0.     (3)
    n=0

  This expression assumes that b'(n,k) = 0 for n < 0 or n >= k and for k < 1.
  Noting that x=py+q, (p=1/c, q=-d/c) and 2y T(n,y)=T(n+1,y)+T(n-1,y),

    k-1
    SUM [ b'(n,k) - p b'(n-1,k-1) - p b'(n+1,k-1) - 2q b'(n,k-1)        (4)
    n=0
          + b'(n,k-2) ] T(n,y) - p b'(0,k-1) T(1,y) - a(N-k) T(0,y) = 0.

  The coefficients of the Chebyshev polynomials must be zero, giving

    b'(n,k) = 2q b'(n,k-1) + p b'(n-1,k-1) + p b'(n+1,k-1) - b'(n,k-2)
              + p b'(0,k-1) D(1,n) + a(N-k) D(0,n),                     (5)

  where D(n,m)=1 if n=m, and 0 otherwise.  This gives the coefficients of the
  expansion for S'(k,x), but what we really want are the coefficients of the
  expansion of S(k,x) at k = N,

             N-1                 N-1
    S(N,x) = SUM b(n,N) T(n,y) = SUM a(n) T(n,x).
             n=0                 n=0

  Using (2), we get

    b(n,N) = b'(n,N) - x b'(n,N-1).

  This then can be substituted in (3), and further manipulated to give an
  expression for b(n,k) which is the same as (5), but with p replaced by p/2
  and q replaced by q/2.

  The coefficients b(n,N) are calculated from the coefficients a(n) as follows.
  (a) Calculate b'(n,k), 0 <= n < k using the recursion (5) for 1 <= k <= N-1.
  (b) For k = N, calculate b(n,N), 0 <= n < N using recursion (5), but with
      p replaced by p/2 and q replaced by q/2.

  With b'(n,k) = 0 for n < 0 or n >= k and k < 1.  The terms on the righthand
  side of (4) have the following ranges.
    2q b'(n,k-1)          0 <= n <= k-2,
    p b'(n-1,k-1)         1 <= n <= k-1,
    p b'(n+1,k-1)         0 <= n <= k-3,
    -b'(n,k-2)            0 <= n <= k-3,
    b'(0,k-1) D(1,n)      n = 1,
    a(N-k) D(0,n)         n = 0.
*/
/* Define the parameters of the inverse transformation */
    p = 1.0 / c;
    q = -d / c;

/*  Find the coefficients of the transformed Chebyshev series */
    for (k = 1; k <= Ncof; ++k) {

      if (k == Ncof) {
	p *= 0.5;
	q *= 0.5;
      }

      /* Update bnp <==> b'(n,k-1) and bnpp <==> b'(n,k-2) */
      for (i = 0; i <= k - 3; ++i) {
	bnpp[i] = bnp[i];
	bnp[i] = bcof[i];
      }
      if (k >= 2)
	bnp[k-2] = bcof[k-2];

      for (n = 0; n < k; ++n) {
	if (n > 1)
	  bb = p * bnp[n-1];
	else if (n == 0)
	  bb = acof[Ncof-k];
	else		/* n == 1 */
	  bb = 2.0 * p * bnp[0];
	if (n <= k - 3)
	  bb = bb - bnpp[n] + p * bnp[n+1] + 2.0 * q * bnp[n];
	else if (n <= k - 2)
	  bb = bb + 2.0 * q * bnp[n];
	bcof[n] = bb;
      }
    }

/* Deallocate the temporary array space */
    UTfree (bnp);
    UTfree (bnpp);
  }
}
