/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  struct Bspec *MMreadSpec (int Ftype, int Ncof, double Sfreq)

Purpose:
  Read FIR filter design specifications

Description:
  This routine prompts for and reads specifications for a filter design.

Parameters:
  <-  struct Bspec *MMreadSpec
      Band specifications.  The space for this structure is allocated
      internally by this routine.  This space can be recovered by invoking
      UTfree.
   -> int Ftype
      Filter type
   -> int Ncof
      Number of filter coefficients
   -> double Sfreq
      Sampling frequency

Author / revision:
  P. Kabal  Copyright (C) 1995
  $Revision: 1.8 $  $Date: 1995/11/24 13:26:55 $


-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: MMreadSpec.c 1.8 1995/11/24 FilterDesign-V1R7a $";

#include <stdio.h>
#include <libtsp.h>
#include <libtsp/nucleus.h>
#include "DFiltFIR.h"

#define MIDBAND		0	/* Mid-band */
#define TRANS		1	/* Transition */

struct Bspec *
MMreadSpec (Ftype, Ncof, Sfreq)

     int Ftype;
     int Ncof;
     double Sfreq;

{
  int state;
  char prompt[100];
  char *line;
  float rval[5];
  float fp;
  int Nb, Nv, i, n;
  struct Bspec *S;
  int sym, even;

/* Check the filter type specification */
  if (Ftype != BPF && Ftype != REC && Ftype != DIF && Ftype != HIL)
    UThalt ("MMreadSpec: Invalid filter type");

/* Initialization */
  sym = (Ftype == BPF || Ftype == REC);
  even = ((Ncof % 2) == 0);

/* Allocate the band specification structure */
  S = (struct Bspec *) UTmalloc (sizeof (struct Bspec));
  
/* Read the band specifications */
  fp = -1.0;
  Nb = 0;
  state = TRANS;
  i = 0;
  Nv = 0;
  while (i <= MAXSPEC) {

/* Prompt and read */
    if (Ftype != DIF)
      sprintf (prompt, "Band %d: Freq. Value Weight [Low High]\n", Nb + 1);
    else
      sprintf (prompt, "Band %d: Freq. Slope Weight [Low High]\n", Nb + 1);
    if (state != MIDBAND)
      FLprompt (prompt);
    line = FLreadLine ("  Spec: ");

    if (line == NULL || (state != MIDBAND && *(STtrimIws (line)) == '\0'))
      break;

/* Check for a transition band specification */
    if (*(STtrimIws (line)) == '\0') {
      if (state == MIDBAND) {
        S->Nval[Nb++] = Nv;
	Nv = 0;
      }
      state = TRANS;
      continue;
    }

/* Specification line */
    if (i >= MAXSPEC)
      UThalt ("%s: Too many specification lines", PROGRAM);
    rval[3] = NOLIMIT_L;
    rval[4] = NOLIMIT_U;
    if (STdecNfloat (line, 3, 5, rval, &n) != 0)
      UThalt ("%s: Invalid specification line", PROGRAM);
    S->f[i] = rval[0] / Sfreq;
    S->val[i] = rval[1];
    S->w[i] = rval[2];
    S->liml[i] = rval[3];
    S->limu[i] = rval[4];

/* Error checks */
    if (S->f[i] <= fp)
      UThalt ("%s: Frequency values not in increasing order", PROGRAM);
    if (S->f[i] < 0 || S->f[i] > 0.5)
      UThalt ("%s: Invalid frequency", PROGRAM);
    if (S->w[i] <= 0.0)
      UThalt ("%s: Non-positive weight value", PROGRAM);
    if (S->liml[i] > S->val[i] || S->limu[i] < S->val[i])
      UThalt ("%s: Invalid constraint value", PROGRAM);

/*
   Check the endpoint specifications
   1. Hilbert transforms must be zero at zero frequency
   2. Symmetrical filters (BPF or REC) of even length and anti-symmetrical
      filters (HIL or DIF) of odd length must be zero at frequency 0.5
*/
    if (S->f[i] == 0.0 && Ftype == HIL && S->val[i] != 0.0)
      UThalt ("%s: Unallowed response at zero frequency", PROGRAM);
    if (S->f[i] == 0.5 && sym == even && S->val[i] != 0.0)
      UThalt ("%s: Unallowed value at half sampling rate", PROGRAM);

    state = MIDBAND;
    ++Nv;

/* Check for end of specifications */
    if (S->f[i] >= 0.5) {
      S->Nval[Nb++] = Nv;
      break;
    }
    fp = S->f[i];
    ++i;
  }

/* Return the specifications to the caller */
  if (Nb == 0)
    UThalt ("%s: No band specifications", PROGRAM);
  S->Nbands = Nb;

  return S;
}
