/*------------- Telecommunications & Signal Processing Lab -------------
                           McGill University

Routine:
  void MMoptions (int argc, const char *argv[], int *Ftype, int *Ncof,
                  float *Sfreq, float *Gridd, const char **Fname)

Purpose:
  Decode options for DFiltFIR

Description:
  This routine decodes filter design options for DFiltFIR.

Parameters:
   -> int argc
      Number of command line arguments
   -> const char *argv[]
      Array of pointers to argument strings
  <-  int *Ftype
      Filter type
  <-  int *Ncof
      Number of filter coefficients
  <-  float *Sfreq
      Sampling frequency
  <-  float *Gridd
      Grid density
  <-  const char **Fname
      Coefficient file name, default NULL

Author / revision:
  P. Kabal  Copyright (C) 1995
  $Revision: 1.7 $  $Date: 1995/05/26 00:22:32 $

----------------------------------------------------------------------*/

static char rcsid[] = "$Id: MMoptions.c 1.7 1995/05/26 FilterDesign-V1R7a $";

#include <stdlib.h>		/* prototype for exit */
#include <stdio.h>
#include <libtsp.h>
#include "DFiltFIR.h"

#ifndef EXIT_SUCCESS
#  define EXIT_SUCCESS	0	/* Normally in stdlib.h */
#endif

#define ERRSTOP(text,par)	UThalt ("%s: %s: \"%s\"", PROGRAM, text, par)

/* Option tables and usage message */
#define LOPT	((sizeof OptTable) / (2 * (sizeof OptTable[0])))
static const char *nullTable[] = { NULL };
static const char *OptTable[] = {
  "-t#", "--t*ype=",
  "-n#", "--n*umber_coefficients=",
  "-s#", "--s*rate=",
  "-g#", "--g*rid_density=",
  "-h", "--h*elp",
  "-v", "--v*ersion",
  "--",
  NULL
};
static const char Usage[] = "\
Usage: %s [options] [FiltFile]\n\
Options:\n\
  -t TYPE, --type=TYPE        Filter type, default bpf\n\
                              \"bpf\", \"receive\", \"differentiator\", or\n\
                              \"hilbert_transform\"\n\
  -n NCOF, --number_coefficients=NCOF\n\
                              Number of filter coefficients.\n\
  -s SFREQ, --srate=SFREQ     Sampling frequency (of the input sequence).\n\
  -g NGRID, --grid_density=NGRID  Grid density.\n\
  -h, --help                  Print a list of options and exit.\n\
  -v, --version               Print the version number and exit.";

void
MMoptions (argc, argv, Ftype, Ncof, Sfreq, Gridd, Fname)

     int argc;
     const char *argv[];
     int *Ftype;
     int *Ncof;
     float *Sfreq;
     float *Gridd;
     const char **Fname;

{
  static const char *TypeTab[] =
    {"b*pf", "r*eceive", "d*ifferentiator", "h*ilbert_transform", NULL};
  int Index;
  const char *OptArg;
  const char **optt;

  int nF;
  int ftype;
  int ncof;
  float sfreq;
  float gridd;
  int n, nn;

/* Defaults */
  ftype = BPF;
  sfreq = 1;
  gridd = GRIDD_DEF;
  *Fname = NULL;

/* Initialization */
  UTsetProg (PROGRAM);
  nF = 0;
  ncof = 0;

/* Decode options */
  Index = 1;
  optt = OptTable;
  while (Index < argc) {
    n = UTgetOption (&Index, argc, argv, optt, &OptArg);
    nn = ((n + 3) / 2) - 1;	/* n = -2 ==> nn = -1 */
    switch (nn) {
    case 0:
      /* Coefficient file */
      ++nF;
      if (nF > 1)
        UThalt ("%s: Too many filenames specified", PROGRAM);
      *Fname = OptArg;
      break;
    case 1:
      /* Filter type */
      ftype = STkeyMatch (OptArg, TypeTab) + 1;
      if (ftype <= 0)
	ERRSTOP ("Invalid filter type specification", OptArg);
      break;
    case 2:
      /* Number of coefficents */
      if (STdec1int (OptArg, &ncof) || ncof <= 0)
	ERRSTOP ("Invalid number of coeffients", OptArg);
      break;
    case 3:
      /* Sampling frequency */
      if (STdec1float (OptArg, &sfreq) || sfreq <= 0.0)
	ERRSTOP ("Invalid sampling frequency", OptArg);
      break;
    case 4:
      /* Grid density */
      if (STdec1float (OptArg, &gridd) || gridd <= 0.0)
        ERRSTOP ("Invalid grid density", OptArg);
      break;
    case LOPT-2:
      /* Help */
      UTwarn (Usage, PROGRAM);
      exit (EXIT_SUCCESS);
      break;
    case LOPT-1:
      /* Version */
      printf ("%s: %s\n", PROGRAM, VERSION);
      exit (EXIT_SUCCESS);
      break;
    case LOPT:
      /* Stop interpreting options */
      optt = nullTable;
      break;
    default:
      /* Option error */
      UThalt (Usage, PROGRAM);
      break;
    }
  }

/* Checks, add defaults */
  if (ncof <= 0) {
    UTwarn ("%s - Number of coefficients not specified", PROGRAM);
    UThalt (Usage, PROGRAM);
  }

/* Set return values */
  *Ftype = ftype;
  *Ncof = ncof;
  *Sfreq = sfreq;
  *Gridd = gridd;
}
