/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  void MMmodSpec (int Ftype, const struct Gspec *G, struct Gspec *Gt)

Purpose:
  Modify the specifications for an FIR filter

Description:
  This routine modifies the specifications of certain types of filters.  The
  specifications are given in terms of a desired value, weight and constraints
  at given frequencies.  For receiving filters, the specifications are modified
  to include an x/sin(x) factor.  For differentiators, the specifications are
  changed from a desired slope to a desired value.

Parameters:
   -> int Ftype
       Filter type.
       1 - Multiple passband/stopband filter
       2 - Multiple passband/stopband filter (sin(X)/X compensation)
       3 - Differentiator
       4 - Hilbert transform filter
   -> struct Gspec *G
      Structure with the arrays specifying the filter specifications on the
      frequency grid.
  <-  struct Gspec *Gt
      Structure with the arrays specifying the modified filter specifications
      on the frequency grid.

Author / revision:
  P. Kabal  Copyright (C) 1995
  $Revision: 1.4 $  $Date: 1995/05/26 00:22:32 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: MMmodSpec.c 1.4 1995/05/26 FilterDesign-V1R7a $";

#include <math.h>
#include "DFiltFIR.h"

#ifndef PI		/* Sometimes in math.h */
#  define PI		3.14159265358979323846
#endif

void
MMmodSpec (Ftype, G, Gt)

     int Ftype;
     const struct Gspec *G;
     struct Gspec *Gt;

{
  int i;
  double ch, fr;
  
  Gt->Ngrid = G->Ngrid;

/* For differentiators, change the desired slope to a desired value */
  if (Ftype == DIF) {
    for (i = 0; i <G->Ngrid; ++i) {
      fr = G->grid[i];
      Gt->des[i] = fr * G->des[i];
      Gt->wt[i] = G->wt[i] / fr;
      Gt->liml[i] = fr * G->liml[i];
      Gt->limu[i] = fr * G->limu[i];
      }
  }
    
/* For receive filters, modify the desired value by x/sin(x) */
  else if (Ftype == REC) {
    for (i = 0; i < G->Ngrid; ++i) {
      fr = G->grid[i];
      if (fr != 0.0) {
	ch = fr * PI / sin (fr * PI);
	Gt->des[i] = ch * G->des[i];
	Gt->liml[i] = ch * G->liml[i];
	Gt->limu[i] = ch * G->limu[i];
      }
      else {
	Gt->des[i] = G->des[i];
	Gt->liml[i] = G->liml[i];
	Gt->limu[i] = G->limu[i];
      }
      Gt->wt[i] = G->wt[i];
    }
  }

/* Other types of filters: no change */
  else {
    for (i = 0; i < G->Ngrid; ++i) {
      Gt->grid[i] = G->grid[i];
      Gt->des[i] = G->des[i];
      Gt->wt[i] = G->wt[i];
      Gt->liml[i] = G->liml[i];
      Gt->limu[i] = G->limu[i];
    }
  }
}
