/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  double MmmDev (const double ad[], int Next, const struct Gspec *G,
                 const int Ext[], const int Etype[], int *ier)

Purpose:
  Calculate the deviation for a minimax approximation

Description:
  This routine calculates the deviation for a minimax approximation.  The input
  is a set of desired values, weights and limits at Next points.  This routine
  solves for the deviation that allows for a Next-1 term polynomial to
  alternate about the desired values, with equal (weighted) deviation at the
  given abscissa values.

Parameters:
  <-  float MMmmDev
      Deviation at the design points
   -> const double ad[]
      Array of Lagrange interpolation coefficients (Next values)
   -> int Next
      Number of extremal values
   -> const struct Gspec *G
      Structure with the arrays specifying the filter specifications on the
      frequency grid.
   -> const int Ext[]
      Indices of the alternation points (Next values).  These values must be
      between 0 and G->Ngrid-1.
   -> const int Etype[]
      Type of extremum (Next values).  Etype[i] takes on one of the following
      values,
      -2 - Lower constraint limit
      -1 - Lower ripple
      +1 - Upper ripple
      +2 - Upper constraint limit
  <-  int *ier
      Error parameter.
      0 - No error
      3 - Constraints too tight for the given filter order

Author / revision:
  P. Kabal  Copyright (C) 1995
  $Revision: 1.3 $  $Date: 1995/02/09 22:11:42 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: MMmmDev.c 1.3 1995/02/09 FilterDesign-V1R7a $";

#include <libtsp.h>
#include "DFiltFIR.h"

double
MMmmDev (ad, Next, G, Ext, Etype, ier)

     const double *ad;
     int Next;
     const struct Gspec *G;
     const int Ext[];
     const int Etype[];
     int *ier;

{
  double den, num;
  float devs;
  int i, k;

/* Calculate the deviation:
   Given a set of Next distinct points, an Next-1 coefficient polynomial can
   give an error curve which exhibits equal amplitude alternations (in a
   weighted sense) about the desired value.  In such a formulation, the free
   parameters are the Next-1 coefficients and the deviation giving a total of
   Next linear equations.  The strategy is to solve for the unique value of the
   deviation which allows for this alternation of the error, without solving
   directly for the coefficients of the polynomial.  The deviation is found
   using Cramer's rule.  The determinants in this formulation can be expressed
   in terms of the determinant of a Vandermonde matrix.  The expression for the
   deviation is expressed in terms of the Lagrange coefficients for an Next
   value interpolation.
*/
  num = 0.0;
  den = 0.0;
  for (i = 0; i < Next; ++i) {
    k = Ext[i];
    if (Etype[i] == LDEV) {
      num += ad[i] * G->des[k];
      den += ad[i] / G->wt[k];
    }
    else if (Etype[i] == UDEV) {
      num += ad[i] * G->des[k];
      den -= ad[i] / G->wt[k];
    }
    else if (Etype[i] == LLIM)
      num += ad[i] * G->liml[k];
    else if (Etype[i] == ULIM)
      num += ad[i] * G->limu[k];
    else
      UThalt ("MMmmDev: Invalid limit code");
  }

  if (den == 0.0) {
    UTwarn ("MMmmDev: Constraints too tight");
    *ier = 3;
    devs = DEVMAX;
  }
  else {
    *ier = 0;
    devs = num / den;
  }

  return devs;
}
