/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  struct Gspec *MMgridVal (int Fcase, int Ngridd, const struct Bspec *S)

Purpose:
  Find the filter specifications on a frequency grid

Description:
  Given the input band specifications, this routine generates an equivalent
  specification on a dense array of frequency points.  The intermediate values
  are generated by interpolating using a piecewise monotonic cubic function
  between the values of the input specifications.

Parameters:
   <-  struct Gspec *MMgridVal
      Structure with the arrays specifying the filter specifications on the
      frequency grid.  This structure is allocated by this routine.  The
      structure contains pointers to the arrays.  These are also allocated by
      this routine.  If the structure is G, the memory for G and the arrays it
      points to can be deallocated with UTfree(G->grid) and UTfree(G).
  -> int Fcase
      Type of filter
      1 - even number of coefficients, bandpass filter
      2 - odd number of coefficients, bandpass filter
      3 - even number of coefficients, differentiator or Hilbert transform
      4 - odd number of coefficients, differentiator or Hilbert transform
   -> int Ngridd
      Desired number of grid points.  This value is used to determine the
      maximum spacing between grid values.  The grid points are chosen to
      occur at each specified frequency and uniformly in between these values.
      The actual number of grid points is greater than or equal to Ngridd,
      exceeding Ngridd by less than the number of specified frequencies.
   -> struct Bspec *S
      Band specifications

Author / revision:
  P. Kabal  Copyright (C) 1995
  $Revision: 1.4 $  $Date: 1995/11/24 01:03:49 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: MMgridVal.c 1.4 1995/11/24 FilterDesign-V1R7a $";

#include <math.h>
#include <libtsp.h>
#include "DFiltFIR.h"

/* Internal routine prototypes */
static int
MM_bandVal p_((const float f[], int Nf, double fmin, double fmax,
	       double delf, float fgrid[]));
static int
MM_Ngrid p_((const float f[], int Nf, double fmin, double fmax, double delf));


#define MINV(a, b)	(((a) < (b)) ? (a) : (b))
#define MAXV(a, b)	(((a) > (b)) ? (a) : (b))

struct Gspec *
MMgridVal (Fcase, Ngridd, S)

     int Fcase;
     int Ngridd;
     const struct Bspec *S;

{
  int i, k;
  int Nf, n;
  double Fs, delf, fmin, fmax;
  int Npt;
  struct Gspec *G;

/* Find the grid spacing.  An upper bound on the grid spacing is found by
   dividing the frequency extent of the bands specified by the desired number
   of grid points.
*/
  Fs = 0.0;
  i = 0;
  for (k = 0; k < S->Nbands; ++k) {
    Nf = S->Nval[k];
    Fs += (S->f[i+Nf-1] - S->f[i]);
    i += Nf;
  }
  delf = Fs / Ngridd;

/*
   For cases 2 and 3 problems occur at the maximum frequency (desired value
   divided by zero but with zero weight), while for cases 3 and 4 similar
   problems occur at zero frequency.
*/
  switch (Fcase) {
  case 1:
    fmin = 0.0;
    fmax = 0.5;
    break;
  case 2:
    fmin = 0.0;
    fmax = 0.5 - delf;
    break;
  case 3:
    fmin = delf;
    fmax = 0.5 - delf;
    break;
  case 4:
    fmin = delf;
    fmax = 0.5;
    break;
  default:
    UThalt ("MMgridVal: Invalid case");
  }

/* Find the actual number of grid points.  Additional points occur at each
   specified frequency.  Between specified frequencies, grid points are
   uniformly spaced such that the spacing is at most delf.
*/
  n = 0;
  i = 0;
  for (k = 0; k < S->Nbands; ++k) {
    Nf = S->Nval[k];
    Npt = MM_Ngrid (&S->f[i], Nf, fmin, fmax, delf);
    i = i + Nf;
    n = n + Npt;
  }
  if (n < Ngridd || n >= Ngridd + i)
    UThalt ("MMgridVal: Consistency check failed");

/* Allocate the arrays */
  G = (struct Gspec *) UTmalloc (sizeof (struct Gspec));
  G->Ngrid = n;
  G->grid = (float *) UTmalloc (5 * G->Ngrid * sizeof (float));
  G->des = G->grid + G->Ngrid;
  G->wt = G->des + G->Ngrid;
  G->liml = G->wt + G->Ngrid;
  G->limu = G->liml + G->Ngrid;

/* Calculate the desired response and the weight on the frequency grid.
   Sample points are uniformly spaced between the specified frequencies with a
   frequency separation which is at most delf.
*/
  n = 0;
  i = 0;
  for (k = 0; k < S->Nbands; ++k) {
    Nf = S->Nval[k];
    Npt = MM_bandVal (&S->f[i], Nf, fmin, fmax, delf, &G->grid[n]);

/* Monotonic cubic interpolation of the values and weights */
    MSintMC (&S->f[i], &S->val[i], Nf, &G->grid[n], &G->des[n], Npt);
    MSintMC (&S->f[i], &S->w[i], Nf, &G->grid[n], &G->wt[n], Npt);

/* Interpolate limits; unspecified limits are large values */
    MSintMC (&S->f[i], &S->liml[i], Nf, &G->grid[n], &G->liml[n], Npt);
    MSintMC (&S->f[i], &S->limu[i], Nf, &G->grid[n], &G->limu[n], Npt);

    i = i + Nf;
    n = n + Npt;
  }
  if (n != G->Ngrid)
    UThalt ("MMgridVal: Consistency check failed");

  return G;
}

static int
MM_Ngrid (f, Nf, fmin, fmax, delf)

     const float f[];
     int Nf;
     double fmin;
     double fmax;
     double delf;

{
  int j, n, Ns;
  float fl, fu;

  n = 0;
  fu = MINV (MAXV (f[0], fmin), fmax);
  for (j = 1; j < Nf; ++j) {
    if (f[j] < f[j-1])
      UThalt ("MMgridVal: Frequency values not in increasing order");
    fl = fu;
    fu = MINV (MAXV (f[j], fmin), fmax);

    Ns = ceil ((fu - fl) / delf);
    if (Ns <= 0)
      if (fl == fmin)
	UThalt ("MMgridVal: Frequency too close to zero");
      else if (fu == fmax)
	UThalt ("MMgridVal: Frequency too close to half sampling frequency");
    else
      UThalt ("MMgridVal: Frequencies too closely spaced");
    n += Ns;
  }
  ++n;

  return n;
}

static int
MM_bandVal (f, Nf, fmin, fmax, delf, fgrid)

     const float f[];
     int Nf;
     double fmin;
     double fmax;
     double delf;
     float fgrid[];

{
  int k, j, n;
  float fl, fu;
  int Ns;
  float dx;

  n = 0;
  fu = MINV (MAXV (f[0], fmin), fmax);
  for (j = 1; j < Nf; ++j) {

    /* Set the frequency limits for the interval */
    fl = fu;
    fu = MINV (MAXV (f[j], fmin), fmax);

    /* Calculate Ns, the number of samples for the interval [fl,fu).  The point
       at fu is picked up in the next interval. */
    Ns = ceil ((fu - fl) / delf);
    dx = (fu - fl) / Ns;
    for (k = 0; k < Ns; ++k)
      fgrid[n++] = k * dx + fl;
  }
  fgrid[n++] = fu;

  for (k = 1; k < n; ++k) {
    if (fgrid[k] <= fgrid[k-1])
      UThalt ("MMgridVal: Grid density too high; grid points not distinct");
  };

  return n;
}
