/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  void MMfiltCof (int Fcase, const double alpha[], double h[], int Ncof)

Purpose:
  Find the direct form coefficients for a linear phase FIR filter

Description:
  This routine finds the coefficents of a linear phase FIR filter given the
  coefficients of a canonical response in terms of a cosine expansion.  Four
  different cases are considered (Ncof even or odd, and h symmetric or anti-
  symmetric).

Parameters:
   -> int Fcase
      Filter case.
      1 - odd number of coefficients, bandpass filter
      2 - even number of coefficients, bandpass filter
      3 - odd number of coefficients, differentiator or Hilbert transform
      4 - even number of coefficients, differentiator or Hilbert transform
   -> const double alpha[]
      Coefficients of the canonical response.  The number of coefficients M is
      uniquely related to Ncof, the number of filter coefficients.
      Fcase=1, M=(Ncof-1)/2+1
            2, M=Ncof/2
            3, M=(Ncof-1)/2-1
            4, M=Ncof/2
  <-  double h[]
      Array of resultant filter coefficients (Ncof values)
   -> int Ncof
      Number of filter coefficients

Author / revision:
  P. Kabal  Copyright (C) 1995
  $Revision: 1.3 $  $Date: 1995/02/09 22:09:16 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: MMfiltCof.c 1.3 1995/02/09 FilterDesign-V1R7a $";

#include <libtsp.h>
#include "DFiltFIR.h"

void
MMfiltCof (Fcase, alpha, h, Ncof)

     int Fcase;
     const double alpha[];
     double h[];
     int Ncof;

{
  int i, m;

/*
   Converting linear phase filters to a canonical form:

   Let Z(n)=z^n .  The zero phase form of the filter response is

            N-1
     H(z) = SUM h(n) Z(-n+(N-1)/2) .
            n=0

  This filter has only M, M < N, degrees of freedom due to the symmetries
  imposed on the coefficients.  We will express H(z) as

    H(z) = P(z) C(z) ,

  where C(z) is (zero-phase) symmetric, C(z)=C(1/z),

            M-1
    C(z) =  SUM  c(n) Z(-n) , with c(n)=c(-n) , and
          n=-(M-1)

  P(z) is a term which is independent of the filter coefficients.  The
  structure of P(z) can be implied from the symmetries of H(z).  Express C(z)
  as

    C(z) = [A(z) + A(1/z)]/2 ,

         = 0.5 a(-n) ,  n=-(M-1),...,-1
    c(n) = a(0) ,       n=0
         = 0.5 a(n) ,   n=1,2,...,M-1 .
  Then
                M-1
    H(w) = P(w) SUM a(n) cos(wn) .
                n=0
*/

/*----------
 Bandpass filter: odd N, symmetric coefficients
 h(n)=h(N-n-1), M=(N-1)/2+1

                   M-1
   H(z) = h(M-1) + SUM h(M-n-1) [Z(n) + Z(-n)]
                   n=1

        = B(1/z) + B(z) ,

              M-1
 where B(z) = SUM b(n) Z(n) ,  with h(n) = b(M-n-1) , n=0,1,...,M-2 ,
              n=0                        = 2 b(0) ,   n=M-1 .

 In this case, P(z)=1, A(z)=B(z),
   h(n) = 0.5 a(M-n-1) , n=0,1,...,M-2
        = a(0) ,         n=M-1 .
*/

  if (Fcase == 1) {
    m = (Ncof - 1) / 2 + 1;
    for (i = 0; i <= m - 2; ++i)
      h[i] = 0.5 * alpha[m-i-1];
    h[m-1] = alpha[0];
    for (i = m; i < Ncof; ++i)
      h[i] = h[Ncof-i-1];
  }

/*----------
  Bandpass filter: even N, symmetric coefficients
  h(n)=h(N-n-1), M=N/2
 
           M-1
    H(z) = SUM h(M-n-1) [Z(n+1/2) + Z(-n-1/2)]
           n=0

         = Z(-1/2) [z B(1/z) + B(z)] ,

               M-1
  where B(z) = SUM b(n) Z(n) , with h(n) = b(M-n-1) , n=0,1,...,M-1 .
               n=0
  Let
    Q(z) = z B(1/z) + B(z)
         = P'(z) C(z) , where P(z) = Z(-1/2) P'(z) .

  Q(z) has coefficients q(n) for n=-M,...,M-1.

    b(n) = q(n) , n=0,1,...,M-1

  From the definition of Q(z), it can be seen that z=-1 is a root.

    P'(z) = z + 1 and P(z) = Z(1/2) + Z(-1/2) .

  With this choice, Q(z) = z C(z) + C(z) , with q(n) = c(n+1) + c(n) ,
  and C(z) = C(1/z).  Finally

    H(z) = P(z) C(z) and H(w) = 2 cos(w/2) C(w) .

         = 0.5 a(M-1) ,              n=0 ,
    h(n) = 0.5 [a(M-n-1) + a(M-n)] , n=1,...,M-2 ,
         = a(0) + 0.5 a(1) ,         n=M-1 .
*/

  else if (Fcase == 2) {
    m = Ncof / 2;
    if (m > 1) {
      h[0] = 0.5 * alpha[m-1];
      for (i = 1; i <= m - 2; ++i)
        h[i] = 0.5 * (alpha[m-i-1] + alpha[m-i]);
      h[m-1] = alpha[0] + 0.5 * alpha[1];
    }
    else
      h[0] = alpha[0];
    for (i = m; i < Ncof; ++i)
      h[i] = h[Ncof-i-1];
  }

/*----------
  Differentiator or Hilbert transform: odd N, anti-symmetric coefficients
  h(n)=-h(N-n-1), M=(N-1)/2

           M-1
    H(z) = SUM h(M-n-1) [Z(n) - Z(-n)],    h(M-1)=0 ,
           n=1

         = z B(1/z) - Z(-1) B(z) ,

               M-1
  where B(z) = SUM b(n) Z(n) ,  with h(n) = b(M-n-1) , n=0,1,...,M-2 ,
               n=0                        = 2 b(0) ,   n=M-1 .
  Let
    Q(z) = Z(2) B(1/z) - B(z)
         = P'(z) C(z) , where P(z) = Z(2) P'(z) .

  Q(z) has coefficients q(n) for n=-(M+1),...,M-1 (with q(-1)=0).

    b(n) = -q(n) , n=0,1,...,M-1.

  From the definition of Q(z), it can be seen that z=1 and z=-1 are roots.

    P'(z) = Z(2) - 1 , and P(z) = z - Z(-1) .

  With this choice, Q(z) = Z(2) C(z) - C(z) , with q(n) = c(n+2) + c(n) , and
  C(z) = C(1/z).  Finally

    H(z) = P(z) C(z) and H(w) = 2j sin(w) C(w) .

         = 0.5 a(M-1) ,              n=0 ,
    h(n) = 0.5 a(M-2) ,              n=1
         = 0.5 [a(M-n-1)-a(M-n+1)] , n=2,...,M-2 ,
         = a(0) - 0.5 a(2) ,         n=M-1 .
*/

  else if (Fcase == 3) {
    m = (Ncof - 1) / 2;
    if (m >= 3) {
      h[0] = 0.5 * alpha[m-1];
      h[1] = 0.5 * alpha[m-2];
      for (i = 2; i <= m - 2; ++i)
        h[i] = 0.5 * (alpha[m -i-1] - alpha[m-i+1]);
      h[m - 1] = alpha[0] - 0.5 * alpha[2];
    }
    else if (m == 2) {
      h[0] = 0.5 * alpha[1];
      h[1] = alpha[0];
    }
    else if (m == 1)
      h[0] = alpha[0];
    h[m] = 0.0;
    for (i = m + 1; i < Ncof; ++i)
      h[i] = -h[Ncof-i-1];
  }

/*----------
  Differentiator or Hilbert transform: even N, anti-symmetric coefficients
  h(n)=-h(N-n-1), M=N/2
           M-1
    H(z) = SUM h(M-n-1) [Z(n+1/2) - Z(-n-1/2)]
           n=0

         = Z(-1/2) [z B(1/z) - B(z)] ,

               M-1
  where B(z) = SUM b(n) Z(n) , with h(n) = b(M-n-1) , n=0,1,...,M-1 .
               n=0
  Let
    Q(z) = z B(1/z) - B(z)
         = P'(z) C(z) , where P(z) = Z(-1/2) P'(z) .

  Q(z) has coefficients q(n) for n=-M,...,M-1.

    b(n) = -q(n) , n=0,1,...,M-1

  From the definition of Q(z), it can be seen that z=1 is a root.

    P'(z) = z - 1 and P(z) = Z(1/2) - Z(-1/2) .

  With this choice, Q(z) = z C(z) - C(z) , with q(n) = c(n+1) - c(n), and
  C(z) = C(1/z).  Finally

    H(z) = P(z) C(z) , H(w) = 2j sin(w/2) C(w) .

         = 0.5 a(M-1) ,              n=0 ,
    h(n) = 0.5 [a(M-n-1) - a(M-n)] , n=1,...,M-2 ,
         = a(0) - 0.5 a(1) ,         n=M-1 .
*/

  else if (Fcase == 4) {
    m = Ncof / 2;
    if (m > 1) {
      h[0] = 0.5 * alpha[m-1];
      for (i = 1; i <= m - 2; ++i)
        h[i] = 0.5 * (alpha[m-i-1] - alpha[m-i]);
      h[m-1] = alpha[0] - 0.5 * alpha[1];
    }
    else
      h[0] = alpha[0];
    for (i = m; i < Ncof; ++i)
      h[i] = -h[Ncof-i-1];
  }

  else
    UThalt ("MMfiltCof: Invalid case");
}
