/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  void MMcosCof (double xs, double xf, const double x[], const double y[],
                 int Next, double alpha[])

Purpose:
  Generate the sum of cosines coefficients for a filter

Description:
  Given the sample values for a polynomial, this routine returns the
  coefficients of the sum of cosines expansion.  The expansion is of the form

           Ncof-1
    G(w) =  SUM  a(n) cos(wn) , where Ncof = Next-1.
            n=0
   
  The input values are Next values of G(w), y(i)=G(w(i)) and x(i)=cos(w(i)),
  i=0,...,Next-1.

Parameters:
   -> double xs
      Starting abscissa limit.  This value is normally in the interval
      [max(x[i]),+1].  Together with xf it defines the interval over which
      the design was specified.
   -> double xf
      Last abscissa limit.  This value is normally in the interval
      [-1,min(x[i])].  Together with xs it defines the interval over which
      the design was specified.
   -> const double x[]
      Array of Next abscissa values (-1 to +1).  These values must be in
      decreasing order.
   -> const double y[]
      Array of Next ordinate values
   -> int Next
      Number of values
  <-  double alpha[]
      Returned array (Next-1 values) of coefficients for the polynomial that
      passes through the given values

Author / revision:
  P. Kabal  Copyright (C) 1995
  $Revision: 1.4 $  $Date: 1995/05/26 00:21:39 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: MMcosCof.c 1.4 1995/05/26 FilterDesign-V1R7a $";

#include <math.h>
#include <libtsp.h>
#include "DFiltFIR.h"

#define EPS	0.05
#define FSH	1e-6
#define PIx2		6.28318530717958647693

void
MMcosCof (xs, xf, x, y, Next, alpha)

     double xs;
     double xf;
     const double x[];
     const double y[];
     int Next;
     double alpha[];

{
  double sum, c, d;
  int i, k, n, M;
  double *acof;
  double *G;
  double xt;
  int Npar;
  struct Pcev *P;

/* Check input limits */
  if (xs > 1.0 || xs < xf || xf < -1.0)
    UThalt ("MMcosCof: Invalid frequency limits");

/* Allocate arrays */
  Npar = Next - 1;
  P = (struct Pcev *) UTmalloc (sizeof (struct Pcev));
  P->ad = (double *) UTmalloc (Next * sizeof (double));
  P->N = Next;
  P->x = (double *) x;
  P->y = (double *) y;
  acof = (double *) UTmalloc (Npar * sizeof (double));
  G = (double *) UTmalloc (Npar * sizeof (double));

/*
   If the band specifications do not extend from 0 to 1/2 (normalized Hz) or
   +1 to -1 (cos(2 Pi f) domain), the frequency response is not sampled at
   the edges.  The full band samples are w(k) = 2 Pi k/M.  For partial band
   sampling, the samples are taken at c*cos(2 Pi k/M) + d.
*/
  if (xf != xs && (xs - xf) < 2.0 - EPS) {
    c = 0.5 * (xs - xf);	/* Note: xs > xf */
    d = 0.5 * (xf + xs);
  }
  else {
    c = 1.0;
    d = 0.0;
  }

/* Generate the Lagrange interpolation coefficients */
/*
   Note that there are Next values given.  In general an Next term polynomial
   is required to pass throught these points.  However, it is assumed that the
   values are constrainted such that a Next-1 term polynomial can pass through
   the Next points.  We could just use a subset of Next-1 values to do the
   Lagrange interpolation.  However, we opt here to use Next values as was done
   at the design stage,   The frequency response will then be Next-1 samples
   from this (Next term) polynomial.
*/
  MMintCof (x, P->ad, Next);

/*
   Evaluate the function using MMintVal at equally spaced intervals
   While f goes from 0 to 0.5, x goes from +1 to -1.
*/
  M = 2 * Npar - 1;
  n = 0;
  for (k = 0; k < Npar; ++k) {

    xt = c * cos ((PIx2 * k) / M) + d;

/* If the abscissa value lies close to one of the grid values, use the
   corresponding ordinate value to avoid a divide by zero in MMintVal
*/
    while (xt <= x[n] - FSH && n + 1 < Next)
      ++n;
    if (xt < x[n] + FSH && xt > x[n] - FSH)
      G[k] = y[n];
    else
      G[k] = MMintVal (xt, P);
  }

/* Evaluate the inverse discrete Fourier transform to find the cosine series
   coefficients
*/

/*
   The frequency response is

             N-1                 M-1
      G(w) = SUM a(n) cos(wn)  = SUM g(n) exp(jwn)
             n=0                 n=0

   where M = 2N-1,
      g(0) = a(0),                      a(0) = g(0)
      g(n) = a(n)/2  for 1 <= n < N,    a(n) = 2 g(n)  for 1 <= n < N
      g(n) = g(M-n)  for N <= n < M.

   Let G(k) = G(2 Pi k/M), the DFT of {g(0),g(1),...,g(M-1)}.  G(k) = G(M-k).
   The inverse DFT gives

            1           N-1
     g(0) = - [G(0) + 2 SUM G(k)],
            M           k=1

            1            N-1
     g(n) = - [ G(0) + 2 SUM G(k) cos(2 Pi kn/M) ],  0 < n < N.
            M            k=1
*/
  sum = 0.0;
  for (k = 1; k < Npar; ++k)
    sum += G[k];
  acof[0] = (G[0] + 2.0 * sum) / M;
  for (i = 1; i < Npar; ++i) {
    sum = 0.0;
    for (k = 1; k < Npar; ++k)
      sum += G[k] * cos ((PIx2 * k * i) / M);
    acof[i] = 2.0 * (G[0] + 2.0 * sum) / M;
  }

/* The inverse discrete Fourier transform gives the coefficients a(n) of a
   Chebyshev series expansion,

            N-1                N-1
     P(x) = SUM a(n) cos(wn) = SUM a(n) T(n,x) .
            n=0                n=0

  The Chebyshev polynomials are polynomials in x=cos(w).  However what we
  really want are the coefficients of the expansion in y=cos(u),

           N-1                N-1
    G(u) = SUM b(n) cos(un) = SUM b(n) T(n,y).
           n=0                n=0
*/

/*
  Routine MMtrCheb finds the coefficients b(n), given the coefficients a(n) and
  the transformation y=cx+d.  For full band sampling, MMtrCheb copies acof to
  alpha.
*/
  MMtrCheb (acof, Npar, c, d, alpha);

  UTfree (acof);
  UTfree (P->ad);
  UTfree (P);
  UTfree (G);
}
