/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  void MMcanonSpec (int Fcase, const struct Gspec *G, struct Gspec *Gt)

Purpose:
  Transform linear phase FIR design specifications into a canonical form

Description:
  Given the specifications of a linear phase FIR filter on a frequency grid,
  this routine transforms the specifications for the design of the different
  cases (odd/even symmetry, odd/even number of coefficients) into a canonical
  form.

  There are four cases to be considered for a linear phase filter,
            N-1       -n 
    H(z) = SUM h(n) z  .
           n=0
  Case     symmetry      even/odd
   1)    h(n)=h(N-1-n)    N odd     bandpass filter
   2)    h(n)=h(N-1-n)    N even    bandpass filter
   3)    h(n)=-h(N-1-n)   N odd     differentiator/Hilbert transform
   4)    h(n)=-h(N-1-n)   N even    differentiator/Hilbert transform

  The frequency response, H(w) = H(exp(jw)), can be written in the following
  form for each of the cases,
                               M-1
    H(w) = exp(jw(N-1)/2) Q(w) SUM a(n) cos(wn) .
                               n=0

  Case     Q(w)       No. coef. (M)
   1)        1          (N-1)/2+1
   2)    2 cos(w/2)      N/2
   3)    2j sin(w)      (N-3)/2+1
   4)    2j sin(w/2)     N/2
  The given specifications will be converted to an equivalent specification for
  approximation by a sum of cosines.  The phase factors in H(w) do not enter
  into the magnitude  approximation.  However, the fixed factor Q(w) must be
  absorbed into the desired value, constraint values, and weighting.

  In addition, this routine coverts the frequency grid values (f from 0 to 1/2)
  to values of x = cos(2Pi f) (x from +1 to -1).

Parameters:
   -> int Fcase
      Filter case.
      1 - odd number of coefficients, bandpass filter
      2 - even number of coefficients, bandpass filter
      3 - odd number of coefficients, differentiator or Hilbert transform
      4 - even number of coefficients, differentiator or Hilbert transform
   -> const struct Gspec *G
      Structure with the arrays specifying the filter specifications on the
      frequency grid
  <-  struct Gspec *Gt
      Structure with the arrays specifying the modified filter specifications
      on a transformed grid

Author / revision:
  P. Kabal  Copyright (C) 1995
  $Revision: 1.5 $  $Date: 1995/11/24 01:02:53 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: MMcanonSpec.c 1.5 1995/11/24 FilterDesign-V1R7a $";

#include <math.h>
#include <libtsp.h>
#include "DFiltFIR.h"

#ifndef PI		/* Sometimes in math.h */
#  define PI		3.14159265358979323846
#endif
#define PIx2		6.28318530717958647693

void
MMcanonSpec (Fcase, G, Gt)

     int Fcase;
     const struct Gspec *G;
     struct Gspec *Gt;

{
  int Ngrid;
  int i;
  double ch;

  Ngrid = G->Ngrid;
  Gt->Ngrid = Ngrid;

  for (i = 1; i < Ngrid; ++i) {
    if (G->grid[i] <= G->grid[i-1])
      UThalt ("MMcanonSpec: Frequency values not in increasing order");
  }

/*
   Set up a new approximation problem which is equivalent to the original
   problem
*/

/* Bandpass filter (odd number of coefficients): no change */
  if (Fcase == 1) {
    for (i = 0; i < Ngrid; ++i) {
      Gt->des[i] = G->des[i];
      Gt->wt[i] = G->wt[i];
      Gt->liml[i] = G->liml[i];
      Gt->limu[i] = G->limu[i];
    }
  }

/* Bandpass filter (even number of coefficients) */
  else if (Fcase == 2) {
    for (i = 0; i <Ngrid; ++i) {
      ch = 2.0 * cos (PI * G->grid[i]);
      Gt->des[i] = G->des[i] / ch;
      Gt->wt[i] = ch * G->wt[i];
      Gt->liml[i] = G->liml[i] / ch;
      Gt->limu[i] = G->limu[i] / ch;
    }
  }

/* Differentiator or Hilbert transform (odd number of coefficients) */
  else if (Fcase == 3) {
    for (i = 0; i <Ngrid; ++i) {
      ch = 2.0 * sin (PIx2 * G->grid[i]);
      Gt->des[i] = G->des[i] / ch;
      Gt->wt[i] = ch * G->wt[i];
      Gt->liml[i] = G->liml[i] / ch;
      Gt->limu[i] = G->limu[i] / ch;
    }
  }
/* Differentiator or Hilbert transform (even number of coefficients) */
  else if (Fcase == 4) {
    for (i = 0; i < Ngrid; ++i) {
      ch = 2.0 * sin (PI * G->grid[i]);
      Gt->des[i] = G->des[i] / ch;
      Gt->wt[i] = ch * G->wt[i];
      Gt->liml[i] = G->liml[i] / ch;
      Gt->limu[i] = G->limu[i] / ch;
    }
  }

/* Change the frequency axis to cosine values */
  for (i = 0; i < Ngrid; ++i)
    Gt->grid[i] = cos (PIx2 * G->grid[i]);

  for (i = 1; i < Ngrid; ++i) {
    if (Gt->grid[i] >= Gt->grid[i-1])
      UThalt ("MMcanonSpec: Frequency values too closely spaced");
  }
}
