/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  int AFreadData (AFILE *AFp, long int offs, float Fbuff[], int Nreq)

Purpose:
  Read data from an audio file (return float values)

Description:
  This routine returns a specified number of samples from an audio file.  The
  data in the file is converted to float format on output.  The sample data in
  the file is considered to be preceded and followed by zero-valued samples.
  Thus if offs is negative or points to beyond the number of samples in the
  file, the appropriate number of zero-valued samples are returned.  The file
  must have been opened using routine AFopenRead.

  The following program fragment illustrates the use of this routine to read
  overlapping frames of data.  For the simpler case of sequential access to the
  data without overlap, the variable Lmem should be set to zero.

    AFp = AFopenRead (...);
    ...
    Lmem =...
    Lframe =...
    Nadv = Lframe-Lmem;
    offs = -Lmem;
    while (1) {
      Nout = AFreadData (AFp, offs, Fbuff, Lframe);
      offs = offs+Nadv;
      if (Nout == 0)
        break;
      ...
    }

Parameters:
  <-  int AFreadData
      Number of data values transferred from the file.  On reaching the end of
      the file, Nout may be less than Nreq, in which case the last Nout-Nreq
      elements are set to zero.  This value can be used by the calling routine
      to determine when the data from the file has been exhausted.
   -> AFILE *AFp
      Audio file pointer for an audio file opened by AFopenRead
   -> long int offs
      Offset into the file in samples.  If offs is positive, the first value
      returned is offs samples from the beginning of the data.  The file data
      is considered to be preceded by zeros.  Thus if offs is negative, the
      appropriate number of zeros will be returned.  These zeros before the
      beginning of the data are counted as part of the count returned in Nout.
  <-  float Fbuff[]
      Array of floats to receive the Nreq samples
   -> int Nreq
      Number of samples requested.  Nreq may be zero.

Author / revision:
  P. Kabal  Copyright (C) 1996
  $Revision: 1.22 $  $Date: 1996/08/13 17:55:09 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: AFreadData.c 1.22 1996/08/13 AFsp-V2R2 $";

#include <stdio.h>
#include <libtsp.h>
#include <libtsp/nucleus.h>
#include <libtsp/AFpar.h>

#define MINV(a, b)	(((a) < (b)) ? (a) : (b))
#define MAXV(a, b)	(((a) > (b)) ? (a) : (b))

/* Reading routines */
static int
(*AF_Read[NFD])p_((AFILE *AFp, long int offs, float Fbuff[], int Nreq)) =
  {
    NULL,
    AFreadMulaw,
    AFreadAlaw,
    AFreadU1,
    AFreadI1,
    AFreadI2,
    AFreadF4,
    AFreadTA
  };

int
AFreadData (AFp, offs, Fbuff, Nreq)

     AFILE *AFp;
     long int offs;
     float Fbuff[];
     int Nreq;

{
  int i, Nb, Nval, Nout;

/* Check the operation  */
  if (AFp->Op != FO_RO)
    UThalt ("AFreadData: File not opened for read");

/* Fill in zeros at the beginning of data */
  Nb = (int) MAXV (0, MINV (-offs, Nreq)); /* Careful: int Nb, long int offs */
  for (i = 0; i < Nb; ++i) {
    Fbuff[i] = 0.0;
    ++offs;
  }

/* Transfer data from file */
  Nval = (*AF_Read[AFp->Format]) (AFp, offs, &Fbuff[Nb], Nreq - Nb);
  Nout = Nb + Nval;

/* Zeros at the end of the file */
  for (i = Nout; i < Nreq; ++i)
    Fbuff[i] = 0.0;

  return Nout;
}
