/*------------- Telecommunications & Signal Processing Lab -------------
                           McGill University

Routine:
  void RSoptions (int argc, const char *argv[], int *Fformat, long int *Nout,
                  double *Soffs, double *Sratio, double *SfreqO,
                  struct Fspec_T *Fspec, const char **Hinfo,
                  const char **NHparms, const char *Fname[2])

Purpose:
  Decode options for ResampAudio

Description:
  This routine decodes options for ResampAudio.

Parameters:
   -> int argc
      Number of command line arguments
   -> const char *argv[]
      Array of pointers to argument strings
  <-  int *Fformat
      Output file format code
  <-  long int *Nout
      Number of output samples to be calculated
  <-  double *Soffs
      Sampel offset
  <-  double *Sratio
      Sampling rate ratio
  <-  double *SfreqO
      Output sampling rate
  <-  struct Fspec_T *Fspec
      Filter specification structure
  <-  const char **Hinfo
      Header information string, default NULL
  <-  const char **NHparms
      Parameters for headerless input files, default NULL
  <-  const char *Fname[2]
      File names: input audio file and output audio file

Author / revision:
  P. Kabal  Copyright (C) 1996
  $Revision: 1.9 $  $Date: 1996/10/29 16:25:13 $

----------------------------------------------------------------------*/

static char rcsid[] = "$Id: RSoptions.c 1.9 1996/10/29 AFsp-V2R2 $";

#include <stdlib.h>		/* prototype for exit */
#include <stdio.h>
#include <libtsp.h>
#include <libtsp/AFpar.h>
#include "ResampAudio.h"
#include "AO.h"

#ifndef EXIT_SUCCESS
#  define EXIT_SUCCESS	0	/* Normally in stdlib.h */
#endif

#define ERRSTOP(text,par)	UThalt ("%s: %s: \"%s\"", PROGRAM, text, par)
#define NELEM(array)	((sizeof array) / (sizeof array[0]))

/* Option tables and usage message */
#define LOPT	(NELEM (OptTable) / 2)
static const char *nullTable[] = { NULL };
static const char *OptTable[] = {
  "-f#", "--f*ilter_spec=",
  "-s#", "--s*rate=",
  "-i#", "--int*erpolate=",
  "-a#", "--a*lignment=",
  "-n#", "--number_samples=",
  "-D#", "--d*ata_format=",
  "-F#", "--file*_type=",
  "-P#", "--p*arameters=",
  "-I#", "--inf*o=",
  "-h", "--h*elp",
  "-v", "--v*ersion",
  "--",
  NULL
};
static const char Usage[] = "\
Usage: %s [options] -f FiltFile AFileI AFileO\n\
Options:\n\
  -s SFREQ, --srate=SFREQ     Sampling frequency for the output file.\n\
  -i SRATIO, --interpolate=SRATIO  Sampling rate ratio.\n\
  -a OFFS, --alignment=OFFS   Offset for first output.\n\
  -f FPARMS, --filter_spec=FPARMS  Filter parameters,\n\
                              \"file=fname,ratio=Ir,delay=Del\" or\n\
                              \"ratio=Ir,cutoff=Fc,gain=g,atten=A,alpha=a,\n\
                              N=Ncof,span=Wspan,offset=Woffs,write=fname\".\n\
  -n NOUT, --number_samples=NOUT  Number of output samples.\n\
  -D DFORMAT, --data_format=DFORMAT  Data format for the output file,\n\
                              \"mu-law\", \"A-law\", \"unsigned8\", \"integer8\",\n\
                              \"integer16\", \"float\", or \"text\".\n\
  -F FTYPE, --file_type=FTYPE  Output file type, \"AFsp\", \"WAVE\", \"AIFF-C\",\n\
                              \"noheader\", or \"noheader_swap\".\n\
  -P PARMS, --parameters=PARMS  Parameters for headerless input files,\n\
                              \"Format,Start,Sfreq,Swapb,Nchan,ScaleF\".\n\
  -I INFO, --info=INFO        Header information string.\n\
  -h, --help                  Print a list of options and exit.\n\
  -v, --version               Print the version number and exit.";

static struct Fspec_T FspecD = {
  NULL, NULL, IR_DEFAULT, DEL_DEFAULT, FC_DEFAULT, GAIN_DEFAULT, ALPHA_DEFAULT,
  NCOF_DEFAULT, WSPAN_DEFAULT, WOFFS_DEFAULT};

void
RSoptions (argc, argv, Fformat, Nout, Soffs, Sratio, SfreqO, Fspec, Hinfo,
	   NHparms, Fname)

     int argc;
     const char *argv[];
     int *Fformat;
     long int *Nout;
     double *Soffs;
     double *Sratio;
     double *SfreqO;
     struct Fspec_T *Fspec;
     const char **Hinfo;
     const char **NHparms;
     const char *Fname[2];

{
  int Index;
  const char *OptArg;
  const char **optt;

  int nF;
  int Dformat, Ftype;
  double Nv, Dv, sratio, sfreqo, soffs;
  int n, nn;
  long int nout;

/* Defaults */
  Dformat = FD_UNDEF;
  Ftype = FW_UNDEF;
  sratio = -1.0;
  sfreqo = -1.0;
  soffs = 0.0;
  nout = 0;
  *NHparms = NULL;
  *Hinfo = NULL;
  Fname[2] = NULL;
  *Fspec = FspecD;

/* Initialization */
  UTsetProg (PROGRAM);
  nF = 0;

/* Decode options */
  Index = 1;
  optt = OptTable;
  while (Index < argc) {
    n = UTgetOption (&Index, argc, argv, optt, &OptArg);
    nn = ((n + 3) / 2) - 1;	/* n = -2 ==> nn = -1 */
    switch (nn) {
    case 0:
      /* Filename argument */
      ++nF;
      if (nF <= 2)
	Fname[nF-1] = OptArg;
      else
	UThalt ("%s: Too many filenames specified", PROGRAM);
      break;
    case 1:
      /* Filter specs */
      RSfiltSpec (OptArg, Fspec);
      break;
    case 2:
      /* Sampling frequency */
      if (STdec1double (OptArg, &sfreqo) || sfreqo <= 0.0)
	ERRSTOP ("Invalid sampling frequency", OptArg);
      break;
    case 3:
      /* Sampling rate ratio */
      if (STdecDfrac (OptArg, &Nv, &Dv) || Nv <= 0 || Dv <= 0)
	ERRSTOP ("Invalid interpolation ratio", OptArg);
      sratio = Nv / Dv;
      break;
    case 4:
      /* Alignment */
      if (STdecDfrac (OptArg, &Nv, &Dv))
	ERRSTOP ("Invalid alignment offset", OptArg);
      soffs = Nv / Dv;
      break;
    case 5:
      /* Number of samples */
      if (STdec1long (OptArg, &nout) || nout <= 0)
	ERRSTOP ("Invalid number of samples", OptArg);
      break;
    case 6:
      /* Data format */
      Dformat = AOdecDFormat (OptArg);
      break;
    case 7:
      /* File types */
      Ftype = AOdecFType (OptArg);
      break;
    case 8:
      /* Headerless input parameters */
      *NHparms = OptArg;
      break;
    case 9:
      /* Header information string */
      *Hinfo = OptArg;
      break;
    case LOPT-2:
      /* Help */
      UTwarn (Usage, PROGRAM);
      exit (EXIT_SUCCESS);
      break;
    case LOPT-1:
      /* Version */
      printf ("%s: %s\n", PROGRAM, VERSION);
      exit (EXIT_SUCCESS);
      break;
    case LOPT:
      /* Stop interpreting options */
      optt = nullTable;
      break;
    default:
      /* Option error */
      UThalt (Usage, PROGRAM);
      break;
    }
  }

/* Checks, add defaults */
  if (nF < 2)
    UThalt ("%s: Too few files specified", PROGRAM);
  if (sfreqo <= 0.0 && sratio <= 0.0)
    UThalt ("%s: Output sampling rate not specified", PROGRAM);
  if (sfreqo > 0.0 && sratio > 0.0)
    UThalt ("%s: Both sampling frequency and interpolation ratio specified",
	    PROGRAM);

/* Set return values */
  *Fformat = Dformat + Ftype;
  *Nout = nout;
  *Soffs = soffs;
  *SfreqO = sfreqo;
  *Sratio = sratio;

  return;
}
