/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  LPanal [options] -p PFile -c LPFile AFileI [AFileO]

Purpose:
  LPC analysis on a speech file

Description:
  This program does linear prediction analysis on an audio file.  The output
  file is a file of linear prediction coefficients and the residual file.

  The steps involved in forming the linear prediction coefficients are as
  follows.
   1: Preemphasize the input signal (first difference filter)
   2: Window the data
   3: Calculate the autocorrelation coefficients for the windowed data
   4: Calculate the LPC coefficients
   5: Bandwidth expand the LPC coefficients

  The steps involved in forming the residual signal are as follows.
   1: Preemphasize the input signal (first difference filter)
   2: Use the LPC coefficients to form the prediction error residual signal.

Options:
  The command line specifies options and file names.  The last file name
  specifies the output file of the linear prediction residual.  Options specify
  the level of information printed during processing.
  -p PFile, --parameter_file=PFile
      Parameter file.
  -c LPFile, --lpc_file=LPFile
      LPC predictor coefficient file.
  -s, --statistics
      Print frame-by-frame statistics.
  -I INFO, --info=INFO
      Header information string.
  -h, --help
      Print a list of options and exit.
  -v, --version
      Print the version number and exit.

  The analysis parameters are read from the parameter file.
    preemphasis_factor = float  ! preemphasis factor (0 to 1, default 0)
    window_length = int         ! analysis window length
    window_offset = int         ! initial offset of the center of the analysis
                                ! window from the center of the frame (negative
                                ! values mean that the analysis window precedes
                                ! frame, default  0)
    window_type = char          ! window type (Hamming or rectangular, default
                                ! Hamming)
    frame_length = int          ! frame size
    LPC_number = int            ! number of LPC coefficients
    LPC_BW_expansion = float    ! bandwidth expansion factor (1 gives no
                                ! bandwidth expansion, default 1)

  For AFsp output files, the audio file header contains an information string.
    Standard Header Information:
      date:1994/01/25 19:19:39 UTC    date
      user:kabal@aldebaran            user
      program:LPanal                  program name
      parameters:<parameters>         list of parameters
  This information can be changed with the header information string which is
  specified as one of the command line options.  Structured information records
  should adhere to the above format with a named field terminated by a colon,
  followed by numeric data or text.  Comments can follow as unstructured
  information.  For the purpose of this program, records are terminated by
  newline characters.  However in the header itself, the newline characters are
  replaced by nulls.  To place a newline character into the header, escape
  the newline character by preceding it with a '\' character.  If the first
  character of the user supplied header information string is a newline
  character, the header information string is appended to the standard header
  information.  If not, the user supplied header information string replaces
  the standard header information.

Environment variables:
  NOHEADER_AUDIOFILE:
  This environment variable defines the data format for headerless or
  non-standard input audio files.  The string consists of a list of parameters
  separated by commas.  The form of the list is
    "Format, Start, Sfreq, Swapb, Nchan, ScaleF"
  Format: File data format
      The lowercase versions of these format specifiers cause a headerless
      file to be accepted only after checking for standard file headers; the
      uppercase versions cause a headerless file to be accepted without
      checking the file header.
       "undefined"                - Headerless files will be rejected
       "mu-law8" or "MU-LAW8"     - 8-bit mu-law data
       "A-law8" or "A-LAW8"       - 8-bit A-law data
       "unsigned8" or "UNSIGNED8" - offset-binary 8-bit integer data
       "integer8" or "INTEGER8"   - two's-complement 8-bit integer data
       "integer16" or "INTEGER16" - two's-complement 16-bit integer data
       "float32" or "FLOAT32"     - 32-bit floating-point data
       "text" or "TEXT"           - text data
  Start: byte offset to the start of data (integer value)
  Sfreq: sampling frequency in Hz (floating point number)
  Swapb: Data byte swap parameter
       "native" - no byte swapping
       "little-endian" - file data is in little-endian byte order
       "big-endian" - file data is in big-endian byte order
       "swap" - swap the data bytes as the data is read
  Nchan: number of channels
      The data consists of interleaved samples from Nchan channels
  ScaleF: Scale factor
      Scale factor applied to the data from the file
  The default values for the audio file parameters correspond to the following
  string.
      "undefined, 0, 8000., native, 1, 1.0"

  AUDIOPATH:
  This environment variable specifies a list of directories to be searched when
  opening the input audio files.  Directories in the list are separated by
  colons (semicolons for MS-DOS).

Author / version:
  P. Kabal / v1r5a  1996/10/29  Copyright (C) 1996

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: LPanal.c 1.16 1996/10/29 AFsp-V2R2 $";

#include <stdio.h>
#include <stdlib.h>
#include <libtsp.h>
#include <libtsp/AFpar.h>
#include "LPanal.h"

#ifndef EXIT_SUCCESS
#  define EXIT_SUCCESS	0	/* Normally in stdlib.h */
#endif

#define MAXHEADER	1024

int
main (argc, argv)

     int argc;
     const char *argv[];

{
  AFILE *AFpI;
  AFILE *AFpL;
  AFILE *AFpO;
  const char *Hinfo;
  const char *Fname[4];
  char Fn[FILENAME_MAX+1];
  float Sfreq;
  int Fstats;
  long int Nsamp, Nchan;
  const float *Win;
  int Lwin, Woffs, Lframe, Np;
  float pre, bwexp;
  char Hstring[MAXHEADER+1];
  int Fformat;

/* Get the input parameters */
  LPoptions (argc, argv, &Fstats, &Hinfo, Fname);
  Fformat = FD_FLOAT32 + FW_AFSP;

/* Read the analysis parameters */
  LPlpcPar (Fname[0], &pre, &Win, &Lwin, &Woffs, &Lframe, &Np, &bwexp);

/* Open the input audio file */
  AFsetNH ("$NOHEADER_AUDIOFILE");
  FLpathList (Fname[1], "$AUDIOPATH", Fn);
  AFpI = AFopenRead (Fn, &Nsamp, &Nchan, &Sfreq, stdout);
  if (Nchan != 1)
    UThalt ("%s: Multiple input channels not supported", PROGRAM);
  printf ("\n");

/* Set the header information string */
  if (Hinfo != NULL)
    AFsetHinfo (Hinfo);
  STcopyMax (AFgetHinfo (), Hstring, MAXHEADER);	/* Save the header */

/* Open the LPC coefficient file */
  FLbackup (Fname[2]);
  AFpL = AFopenWrite (Fname[2], FD_FLOAT32, Np, Sfreq/Lframe, stdout);

/* Open the output residual file */
  AFsetHinfo (Hstring);
  if (Fname[3] != NULL) {
    FLbackup (Fname[3]);
    AFpO = AFopenWrite (Fname[3], Fformat, 1L, Sfreq, stdout);
  }
  else
    AFpO = NULL;

/* Process the input file */
  LPlpcAnal (AFpI, AFpL, AFpO, Fstats, pre, Win, Lwin, Woffs, Lframe, Np,
	     bwexp);

/* Close the files */
  AFclose (AFpI);
  AFclose (AFpL);
  if (AFpO != NULL)
    AFclose (AFpO);

  return EXIT_SUCCESS;
}
