/*------------- Telecommunications & Signal Processing Lab -------------
                           McGill University

Routine:
  void FAoptions (int argc, const char *argv[], int *Fformat, long int *Nout,
                  long int *idoffs, int *Nsub, int *Ir, const char **Hinfo,
                  const char **NHparms, const char *Fname[3])

Purpose:
  Decode options for FiltAudio

Description:
  This routine decodes options for FiltAudio.

Parameters:
   -> int argc
      Number of command line arguments
   -> const char *argv[]
      Array of pointers to argument strings
  <-  int *Fformat
      Output file format code
  <-  long int *Nout
      Number of output samples to be calculated
  <-  long int *idoffs
      Data offset
  <-  int *Nsub
      Subsampling ratio
  <-  int *Ir
      Interpolating ratio
  <-  const char **Hinfo
      Header information string, default NULL
  <-  const char **NHparms
      Parameters for headerless input files, default NULL
  <-  const char *Fname[3]
      File names: input audio file, output audio file, filter file

Author / revision:
  P. Kabal  Copyright (C) 1996
  $Revision: 1.35 $  $Date: 1996/10/29 16:19:11 $

----------------------------------------------------------------------*/

static char rcsid[] = "$Id: FAoptions.c 1.35 1996/10/29 AFsp-V2R2 $";

#include <limits.h>		/* LONG_MIN */
#include <stdlib.h>		/* prototype for exit */
#include <stdio.h>
#include <libtsp.h>
#include <libtsp/AFpar.h>
#include "FiltAudio.h"
#include "AO.h"

#ifndef EXIT_SUCCESS
#  define EXIT_SUCCESS	0	/* Normally in stdlib.h */
#endif

#define ERRSTOP(text,par)	UThalt ("%s: %s: \"%s\"", PROGRAM, text, par)
#define NELEM(array)	((sizeof array) / (sizeof array[0]))

/* Option tables and usage message */
#define LOPT	(NELEM (OptTable) / 2)
static const char *nullTable[] = { NULL };
static const char *OptTable[] = {
  "-f#", "--f*ilter_file=",
  "-i#", "--int*erpolate=",
  "-a#", "--a*lignment=",
  "-n#", "--number_samples=",
  "-D#", "--d*ata_format=",
  "-F#", "--f*ile_type=",
  "-P#", "--p*arameters=",
  "-I#", "--inf*o=",
  "-h", "--h*elp",
  "-v", "--v*ersion",
  "--",
  NULL
};
static const char Usage[] = "\
Usage: %s [options] -f FiltFile AFileI AFileO\n\
Options:\n\
  -f FiltFile, --filter_file=FiltFile  Filter file name.\n\
  -i IR/NSUB, --interpolate=IR/NSUB  Interpolation ratio.\n\
  -a OFFS, --alignment=OFFS   Offset for first output.\n\
  -n NOUT, --number_samples=NOUT  Number of output samples.\n\
  -D DFORMAT, --data_format=DFORMAT  Data format for the output file,\n\
                              \"mu-law\", \"A-law\", \"unsigned8\", \"integer8\",\n\
                              \"integer16\", \"float\", or \"text\".\n\
  -F FTYPE, --file_type=FTYPE  Output file type, \"AFsp\", \"WAVE\", \"AIFF-C\",\n\
                              \"noheader\", or \"noheader_swap\".\n\
  -P PARMS, --parameters=PARMS  Parameters for headerless input files,\n\
                              \"Format,Start,Sfreq,Swapb,Nchan,ScaleF\".\n\
  -I INFO, --info=INFO        Header information string.\n\
  -h, --help                  Print a list of options and exit.\n\
  -v, --version               Print the version number and exit.";

void
FAoptions (argc, argv, Fformat, Nout, idoffs, Nsub, Ir, Hinfo, NHparms, Fname)

     int argc;
     const char *argv[];
     int *Fformat;
     long int *Nout;
     long int *idoffs;
     int *Nsub;
     int *Ir;
     const char **Hinfo;
     const char **NHparms;
     const char *Fname[3];

{
  int Index;
  const char *OptArg;
  const char **optt;

  int nF;
  int Dformat, Ftype;
  int nsub, ir;
  int n, nn;
  long int offs, nout;

/* Defaults */
  Dformat = FD_UNDEF;
  Ftype = FW_UNDEF;
  nsub = 1;
  ir = 1;
  offs = LONG_MIN;
  nout = 0;
  *NHparms = NULL;
  *Hinfo = NULL;
  Fname[2] = NULL;

/* Initialization */
  UTsetProg (PROGRAM);
  nF = 0;

/* Decode options */
  Index = 1;
  optt = OptTable;
  while (Index < argc) {
    n = UTgetOption (&Index, argc, argv, optt, &OptArg);
    nn = ((n + 3) / 2) - 1;	/* n = -2 ==> nn = -1 */
    switch (nn) {
    case 0:
      /* Filename argument */
      ++nF;
      if (nF <= 2)
	Fname[nF-1] = OptArg;
      else
	UThalt ("%s: Too many filenames specified", PROGRAM);
      break;
    case 1:
      /* Filter file */
      Fname[2] = OptArg;
      break;
    case 2:
      /* Interpolation ratio */
      if (STdecIfrac (OptArg, &ir, &nsub) || ir <= 0 || nsub <= 0)
	ERRSTOP ("Invalid interpolation ratio", OptArg);
      break;
    case 3:
      /* Alignment */
      if (STdec1long (OptArg, &offs))
	ERRSTOP ("Invalid alignment offset", OptArg);
      break;
    case 4:
      /* Number of samples */
      if (STdec1long (OptArg, &nout) || nout <= 0)
	ERRSTOP ("Invalid number of samples", OptArg);
      break;
    case 5:
      /* Data format */
      Dformat = AOdecDFormat (OptArg);
      break;
    case 6:
      /* File types */
      Ftype = AOdecFType (OptArg);
      break;
    case 7:
      /* Headerless input parameters */
      *NHparms = OptArg;
      break;
    case 8:
      /* Header information string */
      *Hinfo = OptArg;
      break;
    case LOPT-2:
      /* Help */
      UTwarn (Usage, PROGRAM);
      exit (EXIT_SUCCESS);
      break;
    case LOPT-1:
      /* Version */
      printf ("%s: %s\n", PROGRAM, VERSION);
      exit (EXIT_SUCCESS);
      break;
    case LOPT:
      /* Stop interpreting options */
      optt = nullTable;
      break;
    default:
      /* Option error */
      UThalt (Usage, PROGRAM);
      break;
    }
  }

/* Checks, add defaults */
  if (nF < 2)
    UThalt ("%s: Too few files specified", PROGRAM);
  if (Fname[2] == NULL)
    UThalt ("%s: No filter file specified", PROGRAM);

/* Set return values */
  *Fformat = Dformat + Ftype;
  *Nout = nout;
  *idoffs = offs;
  *Nsub = nsub;
  *Ir = ir;

  return;
}
