
/*
 *
 *  @(#)ramd.c	3.1 Copyright (C) B.M.Goodheart 1987, 1888, 1989, 1991 
 *
 *
 *
 *  		      DISCLAIMER OF WARRANTY. 
 *  This software is distributed "as is" and without warranties as to 
 *  performance of merchantability or any other warranties whether expressed 
 *  or implied. In no event shall the author (the copyright holder) be held 
 *  liable for any loss of profit or any other commercial damage resulting 
 *  from the use or misuse of this software, including but not limited to 
 *  special, incidental, consequential or other damages.  THE USER MUST
 *  ASSUME THE ENTIRE RISK OF USING THIS SOFTWARE.
 *
 *
 *			LISCENSE AGREEMENT
 *	This software is placed	into the public domain and 
 *	may be copied or distributed freely provided no profit or 
 *	gain is made and is used for personal use only and the code 
 *	as distributed retains all copyright notices in the code. 
 *	This includes any copyright statements in the target
 *	binary and executable code produced from the distributed source. 
 *
 *
 *	                DISTRIBUTION NOTE
 * 	This file contains the code for a RAM disk driver for UNIX System 5.2
 *	and System V.3.	It has been designed specifically for Microport's 
 *	System V/AT iAPX286 and Interactive Systems V/386 i386 systems
 * 	but should work with other PC based System V.? ports with only 
 *	minor modifications, specifically the BUFSEL selector on the 80286.
 *	This driver should work on any V.3 implementation on a PC. 
 *	Thanks to Ron Bolin (gatech!sbmsg1!bsts00!rlb), this driver has
 *	now been ported and tested on ESIX System V.4.0.
 *
 *
 *	               IMPORTANT NOTE !!!
 *	Any files or data stored within the ram disks will be 
 *	LOST FOR EVER if the system is brought down for ANY reason.
 *
 *	There are also two other programs distributed associated 
 *	with the driver, they are ramstat(1M) and raminit(1M). 
 *	These are used to control the ram disk driver in user mode 
 *	See also ram(7). There are also some contibuted support
 *	programs in the contrib directory.
 *
 *	Berny Goodheart (berny@tndsyd.oz@munnari.oz.au)
 *
 * Modifications:
 *	20-10-87 (001)	The ramsize array would still hold the ram
 *			size even if could not get memory. This has
 *			now been fixed, variable is now zerod.
 *
 *	23-10-88 (002)	Reduced ambiguouse code, now drops through
 *			to the same statements. (reduces object size)
 *
 *	23-10-88 (003)	malloc() returns 0 not NULL (who cares)
 *
 *	12-11-88 (004)  fixed a really big bug, god knows how it
 *			worked in the first place. basically, almost
 *			all the routines that made reference to
 *			'dev' as passed to the sysentry call
 *			where using the raw data, now uses minor(dev)
 *			to extraxt the device index proper.
 *
 *	19-06-89 (005)  this mod is huge, basically I have added
 *			a switch to the compiler "i386" which will
 *			create the driver for System V.3. The difference
 *			to V.2 is quite large in that the two systems
 *			have different memory managment, V.3 uses Page Tables
 *			to map physical memory to kernel virtual memory.
 *
 *	20-06-89 (006)  fixed the character read overrun which used to
 *			to panic the system.
 *
 *      07-07-89 (007)  fixed the block count overrun check algorithm in
 *                      the strategy routine.
 *
 *	09-08-89 (008)  Added a few macros so that the code is
 *                      easier to read.
 *
 *	30-11-89 (009)  forgot to add the strategy print routine
 *                      Now the system tells you when you run out of
 *                      space in a ram disk.
 *
 *	15-04-91 (010)  Jan Akalla reported this one. Trying to write to
 *			the last block in a Ram disk gave out-of-space error.
 *
 *	10-05-91 (011)	Ported to SVR4 by Ron Bolin (gatech!sbmsg1!bsts00!rlb)
 *			Atlanta, GA.
 *			Also changed function base names from "ram" to
 *			"ramd". Assume this conflicted with other routines
 *			on System VR4 (B.M.G)
 *
 *			
 */

static char *bmgid = 
#ifdef TEST
     "RAM disk Test Version Copyright (C) Berny Goodheart \n";
#elif SV4
     "RAM disk 3.1-(SV4-%s) Copyright (C) Berny Goodheart 5/20/91\n\n";
#else
     "RAM disk 3.1-(%s) Copyright (C) Berny Goodheart 5/20/91\n\n";
#endif

#include <sys/errno.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/signal.h>
#include <sys/dir.h>
#include <sys/user.h>
#include <sys/file.h>
#include <sys/sysmacros.h>
#include <sys/map.h>
#include <sys/buf.h>
#include <sys/ramd.h>			/* header file for this driver  */

#ifdef SV4
#include <sys/uio.h>
#include <sys/kmem.h>
#define cotb		potb
#define iodone		biodone
struct uio		*uio_p;
#endif


#define DSK_PHYS_BLOCK	512		/* bytes in a physical disk block */

#define devnum(bp)	minor((bp)->b_dev) /* gets the minor device number */
#define blknum(bp)	((bp)->b_blkno)	   /* block offset for I/O */
#define RADDR(d)	(ramadd[minor(d)])
#define ROPEN(d)	(ram_opened[minor(d)])
#define RSTAT(d)	(ram_state[minor(d)])
#define RSIZE(d)	(ramsize[minor(d)])

#ifndef i386 				/* (008) */
#include <sys/mmu.h>			/* for BUFSEL selector */
caddr_t mapin();
#define  raddr_t	paddr_t
#define diskblk(x)	(ctob(x))
#define blktophys(bp)	(ctob(ramadd[devnum(bp)]) + diskblk(blknum(bp)));
static char *machtype = "iAPX286";
#else
#include <sys/immu.h>
#include <sys/cmn_err.h>
#define mapin(a,b)	(a)		/* emulate V.2 MMU stuff */
#define raddr_t		caddr_t
#define diskblk(x)	(x * DSK_PHYS_BLOCK)
#define blktophys(bp)	(ramadd[devnum(bp)] + diskblk(blknum(bp)));
static char *machtype = "iAPX386";
#endif

#ifdef SV4
#include <sys/ddi.h>			/* last include rule for DDK */
#endif

static raddr_t ramadd[NRDEVS];		/* base address of ram disk	*/
static int ram_opened[NRDEVS];		/* number of processes sleeping */
static int ram_state[NRDEVS];		/* current ram state		*/
static int ramDebug = 0;		/* debug static flag, 0 = off   */
static char busywait = 0;		/* sleeping processes           */

/*
 * ramsize[n] = holds the size of ram disk in physical memory blocks
 *
 * NOTE on i386 machines
 *      a physical block must be on a 4k boundry i.e 1 block = 4096 bytes
 * 	on iAPX286 machines a physical block is 512 bytes
 * 
 */
static int ramsize[NRDEVS];		




/* 
 * This routine is called whenever a process calls the system
 * call 'open' on this device
 *
 */
ramdopen(dev)
{

	busywait++;
	(ROPEN(dev))++;

	if(ramDebug) 
		(void)printf("Ram-[%d] opened by uid-[%d]\n"
			,minor(dev)
			,u.u_ruid);

	/*
	 *
	 * Mask out any other opens while we are busy.
	 * The kernel does not experience any further CPU
	 * work while this happens, the process is simply
	 * put to sleep and added to the process que until 
	 * awoken later..
	 *
	 */
	while(busywait != 1)	/* mask out ram_opened opens */
		(void)sleep((caddr_t)&busywait, PRIBIO + 1);

	/* 
	 *
	 * Is it a valid device 
	 *
	 */
	if(minor(dev) < 0 || minor(dev) > (NRDEVS - 1)) 
		u.u_error = ENODEV; /* (002) */

	/*
	 * wakeup any pending I/O
	 *
	 */
	busywait--;
	(void)wakeup((caddr_t)&busywait);
}



ramdclose(dev)
{
	ROPEN(dev) = 0; /* only called on last close */
}


/*
 * strategy routine is called by both character and block
 * drivers to do the I/O.
 */
#ifdef SV4
void
#endif
ramdstrategy(bp)
struct buf *bp;
{
	raddr_t physdr;

	/*
	 *
	 * If we slip up with deletion of the ram
	 * disk for some reason and we get here somehow,
	 * then this check will stop any reads 
	 * or writes to possibly now unassigned memory.
	 *
	 */
	if(ram_state[devnum(bp)] == RAM_CLOSED) {
		bp->b_error = EIO;
		bp->b_flags |= B_ERROR;
		bp->b_resid = bp->b_bcount;
		(void)iodone(bp);
		return;
	}


	/* 
	 *
	 * check its a legal block (007)
	 *
	 * changed >= to > "last block gave no space error" (010)
	 */
#if i386
	if((diskblk(blknum(bp))+bp->b_bcount)>(ramsize[devnum(bp)] * PTSIZE)) {
#else
	if(diskblk(blknum(bp)) > diskblk(ramsize[devnum(bp)])) {
#endif
		if(bp->b_flags & B_READ) 
			bp->b_error = EIO;
		else
			bp->b_error = ENOSPC;
		bp->b_flags |= B_ERROR;
		bp->b_resid = bp->b_bcount;
		(void)iodone(bp);
		return;
	}


	/* 
	 * work out which block number in the mapped in RAM array
	 * and convert to a physical address.
	 */
	physdr = blktophys(bp);



	/*
	 * If debug is set then send out to the console
	 *
	 */
	if(ramDebug) 
		(void)printf("Ram%d: blk-[%ld] addr-[0x%lx] bytes-[%d] %s\n"
			,devnum(bp)
			,blknum(bp)
			,physdr
			,bp->b_bcount
			,bp->b_flags & B_READ ? "Read" : "Write");
	

	/*
	 * Do the physical memory transfer I/O
	 */
	if(bp->b_flags & B_READ) 
		(void)bcopy(mapin(physdr,BUFSEL),bp->b_un.b_addr,bp->b_bcount);
	else 
		(void)bcopy(bp->b_un.b_addr,mapin(physdr,BUFSEL),bp->b_bcount);


	/*
	 * because bcopy() does not return anything
	 * we assume all bytes where copied and there is
	 * none left in the buffer to transmit
	 */
	bp->b_resid = 0;


	/*
	 * All finished, free the buffer
	 */
	(void)iodone(bp);

}




/*
 * ramdread
 *
 * Uses ramstrategy for RAW I/O 
 *
 */
ramdread(dev)
int dev;
{
#ifdef SV4
	physiock(ramdstrategy,NULL,dev,B_READ,RSIZE(dev)*PTSIZE,uio_p);
#else
	(void)physio(ramdstrategy,NULL,dev,B_READ);
#endif
}




/*
 * ramdwrite
 *
 * Uses ramstrategy for RAW I/O 
 *
 */
ramdwrite(dev)
int dev;
{
#ifdef SV4
	physio(ramdstrategy,NULL,dev,B_WRITE,RSIZE(dev)*PTSIZE,uio_p);
#else
	(void)physio(ramdstrategy,NULL,dev,B_WRITE);
#endif
}



/*
 * user interface to I/O control
 *
 */
ramdioctl(dev, cmd, arg)
int dev, cmd;
union r_ramst *arg;
{
	register x;
	int intlevel;
#if i386
#ifndef SV4
	caddr_t ptalloc();	/* page table allocation routine */
#endif

	/*
	 * Only super user can set or unset the
	 * ram disks.
	 *
 	 * NOTE: TCMEMINFO is only available on 386
	 */
	if((!suser()) && ((cmd != TCGETRAM) || (cmd != TCMEMINFO)) ) {
#else
	if((!suser()) && cmd != TCGETRAM) {
#endif
		u.u_error = EACCES;
		return;
	}


	switch(cmd) {
		case  TCSETRAM :
			/*
	 		 * This will usually be done by 'ramstat' (hopefully)
			 *
			 * If already open then return with errno
			 */
			if(RSTAT(dev) == RAM_OPEN) {
				u.u_error = EBUSY;
				return;
			}

			/*
			 * arg is the ram size in physical blocks
			 * with which to create the RAM disk.
			 */
			if(arg->r_arg < MINRAMSIZ) {
				u.u_error = EINVAL;
				return;
			}
			RSIZE(dev) = arg->r_arg;



			/*
			 * turn interrupts off
			 */
			intlevel = spl6();


			/* 
			 * Try to get memory
			 * 
			 */
#if i386
#ifdef SV4
			if((RADDR(dev) = kmem_zalloc(RSIZE(dev)*PTSIZE,
					KM_NOSLEEP)) == 0) {
#else /* SV3 */
			if((RADDR(dev) = ptalloc(RSIZE(dev),
					PHYSCONTIG|NOSLEEP)) == 0) {
#endif
#else
			if((RADDR(dev)=malloc(coremap,RSIZE(dev)))== 0){
#endif
				RSIZE(dev) = 0; /* (001) */
				u.u_error = ENOMEM;
				(void)splx(intlevel);
				return;
			}
#ifndef i386 
			/*
			 *
			 * Clear the Memory clicks to zero's
			 *
			 * not needed on i386
			 */
			for (x = 0; x < RSIZE(dev); x++)
				(void)clearseg(RADDR(dev) + x);
#endif


			/*
			 *
			 * Initialization is complete at this point
			 * so set the ram_state flag.
			 *
			 */
			RSTAT(dev) = RAM_OPEN;

			(void)splx(intlevel);

			break;


		case  TCDELRAM :
			/*
			 * No point in deleting if we don't exist
			 *
			 */
			if(RSTAT(dev) == RAM_CLOSED) {
				u.u_error = ENXIO;
				return;
			}

			/*
			 * Cant delete the RAM disk if busy
			 *
			 */
			if(ROPEN(dev) > 1) {
				u.u_error = EBUSY;
				return;
			} else
				ROPEN(dev) = 0;


			/*
			 * Try to free up the memory
			 * else GOD HELP THE WORLD
			 *
			 */
			intlevel = spl6();
#if i386
			/*
 			 * uptfree() and kmem_free() don't return anything
			 */
#ifdef SV4
			(void)kmem_free(RADDR(dev),RSIZE(dev)*PTSIZE);
#else /* SV3 */
			(void)uptfree(RADDR(dev),RSIZE(dev));
#endif
#else
			if(mfree(coremap,RSIZE(dev), RADDR(dev)) == -1) 
				/*
				 * We have to panic here because if we
				 * can't free up the memmory then anything
				 * can happen. (possible hardware fault).
				 *
				 * praise be the lord 
				 */
				(void)panic("ram%d: memory de-allocation error",
					minor(dev));
#endif


			/*
			 * PHEW ! THAT WAS CLOSE
			 *
			 * Reset the parameters to zilch
			 * so that next initialization is setup ok.
			 *
			 */
			RSTAT(dev) = RAM_CLOSED;
			RADDR(dev) = 0;
			RSIZE(dev) = 0;
			(void)splx(intlevel);
			break;

		case  TCGETRAM :
			/*
			 * Get statistics about the RAM disks
			 *
			 */
			for(x = 0; x < NRDEVS;x++) {
				arg->r_rstat.r_dev[x] = x;
				arg->r_rstat.r_blks[x] = ramsize[x];
				arg->r_rstat.r_stat[x] = 
					ram_state[x] == RAM_OPEN ? 1 : 0 ;
				arg->r_rstat.r_opns[x] = ram_opened[x];
				arg->r_rstat.r_addr[x] = (paddr_t)ramadd[x];
				arg->r_rstat.r_dbg = ramDebug == 0 ? 0 : 1 ;
			}
			break;

		case  TCRAMDBG :
			/*
			 *
			 * Toggle ramDebug mode 
			 *
			 */
			ramDebug = ramDebug == 0 ? 1 : 0;
			arg->r_arg = ramDebug;
			break;

#if i386
		case  TCMEMINFO:
			/*
		 	 * obtain system memory stats
			 *
			 */
			arg->r_meminfo.r_maxmem = maxmem;
			arg->r_meminfo.r_freemem = freemem;
			break;
#endif

		default:
			u.u_error = EINVAL;
	}
}


/*
 * Initialize start up logo.
 * Done at system boot only
 *
 */
ramdinit()
{
	(void)printf(bmgid,machtype);
}


/*
 * Generic kernel print routine 
 * called on internal error in strategy
 * routine (009)
 */
ramdprint(dev,s)
int dev;
char *s;
{
	(void)printf("DANGER: %s on ram disk device %d\n", s, minor(dev));
}
