/* 
 * i550mxmt.c - enqueue a MIP request out to i550 device
 * 
 * Author:	Ralph E. Droms/Thomas Narten
 * 		Dept. of Computer Sciences
 * 		Purdue University
 * Date:	Mon Sep 24 1984
 * Copyright (c) 1984 Ralph E. Droms
 */

#include <i550.h>
#include <kernel.h>

/*
 * i550mxmt
 *	550 MIP Transmit.
 *
 * Inputs:
 *	rqe:	pointer to the request queue entry to be copied to the queue.
 *
 * Outputs:
 *	TRUE:	if the entry is sent
 *	FALSE:	if the queue is full and the entry cannot be sent
 *
 * Calls:
 *	bcopy
 *
 * Called by:
 *	i550rframe
 *	i550tframe
 *
 * This routine copies a request queue entry to the outound request
 * queue to the 550.  An indication of whether the frame got sent or
 * not is returned.  The entry is copied to the request queue and if
 * the copy caused the queue to go from an EMPTY to a NOT_EMPTY state
 * the 550 is signaled of the change in status of the queue.
 *
 * Assumes caller mutexed usage of MIP Q's.
 */
 
i550mxmt(rqe)
register struct mip_rqe *rqe;
{
	register int	saveIndex;
	register int	nextIndex;

	/*
	 * If full, return that fact.
	 */

	if ((i550.XmtQ.d_takeIndex == i550.XmtQ.d_giveIndex)
	&&  ((i550.XmtQ.d_takeState & MIP_TAKE_FACTOR) != (i550.XmtQ.d_giveState & MIP_GIVE_FACTOR)))
		return(FALSE);

	/*
	 * There is room.  Make the entry.
	 */

	bcopy((char *)rqe, (char *)&i550.XmtQ.d_rqe[i550.XmtQ.d_giveIndex], sizeof *rqe);

	/*
	 * Update MIP Queue.
	 * Note: usage of nextState is necessary in MIP protocol to
	 *	 avoid races with receiver use of giveIndex.
	 */

	saveIndex = i550.XmtQ.d_giveIndex;
	nextIndex = (i550.XmtQ.d_giveIndex + 1) % RQD_Q_SIZE;

	if (i550.XmtQ.d_takeIndex == nextIndex)		/* went full */
		i550.XmtQ.d_giveState =
			(i550.XmtQ.d_giveState | MIP_GIVE_FACTOR) &
			~(i550.XmtQ.d_takeState & MIP_TAKE_FACTOR);

	i550.XmtQ.d_giveIndex = nextIndex;

	if (saveIndex == i550.XmtQ.d_takeIndex) {		/* went not-empty */
		i550.XmtQ.d_empty = MIP_Q_NO_LONGER_EMPTY;
		outbyte(I550PORT, I550_START);
	}

	/*
	 * Return indicates success.
	 */

	return(TRUE);
}
