/* 
 * i550mrcv.c - receive frames from MIP queue
 * 
 * Author:	Ralph E. Droms/Thomas Narten
 * 		Dept. of Computer Sciences
 * 		Purdue University
 * Date:	Wed Sep 26 1984
 * Copyright (c) 1984 Ralph E. Droms
 */

#include <i550.h>
#include <kernel.h>

/*
 * i550mrcv
 *	550 MIP Recieve.
 *
 * Inputs:
 *	rqe:	pointer to a field to a request queue entry
 *
 * Outputs:
 *	TRUE:	Got one.
 *	FALSE:	No entry to receive.
 *
 * Calls:
 *	bcopy
 *	out
 *
 * Called by:
 *	i550rframe
 *
 * This routine will fetch the first entry in the inbound request
 * queue from the 550 and copy it to the space provided by the 
 * input parameter.  An indication is returned of whether an entry
 * was present.  If the request queue was full, the 550 is signalled
 * to indicate the change in state of the queue.
 *
 * Assumes caller mutexed usage of MIP Q's.
 */
 
i550mrcv(rqe)
register struct mip_rqe *rqe;
{
	register int	saveIndex;
	register int	nextIndex;

	i550.RcvQ.d_empty = MIP_NO_CHANGE;

	/*
	 * If nothing there, return that fact.
	 */

	if ((i550.RcvQ.d_takeIndex == i550.RcvQ.d_giveIndex)
	&&  ((i550.RcvQ.d_takeState & MIP_TAKE_FACTOR) == (i550.RcvQ.d_giveState & MIP_GIVE_FACTOR)))
		return(FALSE);

	/*
	 * There was something there.  Get it.
	 */

	bcopy((char *)&i550.RcvQ.d_rqe[i550.RcvQ.d_takeIndex], (char *)rqe, sizeof *rqe);

	/*
	 * Update MIP Queue.
	 * Note: usage of nextState is necessary in MIP protocol to
	 *	 avoid races with sender use of takeIndex.
	 */

	saveIndex = i550.RcvQ.d_takeIndex;
	nextIndex = (i550.RcvQ.d_takeIndex + 1) % RQD_Q_SIZE;

	if (i550.RcvQ.d_giveIndex == nextIndex)		/* went empty */
		i550.RcvQ.d_takeState =
			(i550.RcvQ.d_takeState & ~MIP_TAKE_FACTOR) |
			(i550.RcvQ.d_giveState & MIP_GIVE_FACTOR);

	i550.RcvQ.d_takeIndex = nextIndex;

	if (saveIndex == i550.RcvQ.d_giveIndex) {		/* went not-full */
		i550.RcvQ.d_full = MIP_Q_NO_LONGER_FULL;
		outbyte(I550PORT, I550_START);
	}

	/* 
	 * Return indicates we found one.
	 */

	return(TRUE);
}
