/* 
 * i550init.c - boot MIP and initialize i550 device driver
 * 
 * Author:	Ralph E. Droms/Thomas Narten
 * 		Dept. of Computer Sciences
 * 		Purdue University
 * Date:	Mon Sep 24 1984
 * Copyright (c) 1984 Ralph E. Droms
 */

/*
 * i550init
 *	Initialize the 550 for use.
 *
 * Inputs:
 *	none
 *
 * Outputs:
 *	value=	OK if boot OK, SYSERR otherwise
 *
 * Calls:
 *	i550conn
 *	i550sbuf
 *
 * This procedure readies the 550 for use by higher protocols. That is,
 * the device is "configured" and initialized in such a way that it talks to
 * us via interrupts and the mip queues.
 */

#include <i550.h>
#include <conf.h>
#include <kernel.h>


struct i550 i550;	/* control block for i550 status information	*/

i550init(devptr)
struct devsw *devptr;
{
	register struct i550bpres *bp;
	register struct i550bcmd *bc;
	int i550fwv, i550tstr;

	/* 
	 * fill interrupt control vector
	 */

	 ioinit(devptr->dvnum);

	/* 
	 * Create pool of buffers that are passed between the i550
	 * and the 8086. Processes that wish to talk to the
	 * i550 MUST use these buffers, as they are automatically
	 * freed by the lower level i550 routines
	 */

	if ((i550.xmitpid = mkpool((sizeof (struct edl_xmit)), CXMIT_MAX))
	    == SYSERR ) {
	    kprintf("buffer size: %d, number : %d\n",
	    (sizeof(struct edl_sbuf)), CXMIT_MAX);
	    kprintf("i550init: Cannot create buffer pools.\n");
	    return(SYSERR);
	}

	if ((i550.read_sem = screate( 0 )) == SYSERR) {
	    kprintf("i550open: Cannot create read semaphore.\n");
	    return(SYSERR);
	}

	
/*
 * i550boot
 *	See if 550 is there and get it started.
 *
 * This procedure performs a boot of the 550.  It is called during
 * system-initialization to sense the device and report if it's there.
 * It utilizes data passed by the user in the configuration module and
 * internal constants.  It assumes that each request queue has
 * RQD_Q_SIZE entries.  It also assumes only two devices, the 550 being
 * device id '1' (The other device could be a disk). This being device '0'.
 *
 * One large IDS is also assumed, somewhere in the upper 32Kb of memory.
 *
 * The 550 will never be "bootstrapped" again after this procedure is
 * finished.
 */

/*
 * Reset the 550. Reset i550exists flag and the device open flag
 */


	i550.exists = 0;
	i550.dev.d_open = 0;


	outbyte(I550PORT, I550_RESET);
	delay(I550DELAY);		/* "sleep" 2sec */

/*	
 * Do the "presence" command (See if it's there).
 * Only do the rest if this passes.
 */

	bp = (struct i550bpres *)I550BOOT;
	bzero(bp, sizeof *bp);
	bp->bp_command = BOOT_PRESENCE_COMMAND;

	outbyte(I550PORT, I550_START);
	delay(I550DELAY);		/* "sleep" 2sec */

	i550fwv = bp->bp_version;	/* save firmware version # */
	i550tstr = bp->bp_testResult;	/* and test result */

	if (bp->bp_response == BOOT_SUCCESSFUL_RESULT) {
	
		/*
		 * Initialize MIP queues.
		 */
	
		bzero((char *)&i550.XmtQ, sizeof i550.XmtQ);
		bzero((char *)&i550.RcvQ, sizeof i550.RcvQ);
	
		i550.XmtQ.d_rqSize = RQD_Q_SIZE;
		i550.RcvQ.d_rqSize = RQD_Q_SIZE;
		i550.XmtQ.d_rqeLength = LRQE_SIZE;
		i550.RcvQ.d_rqeLength = LRQE_SIZE;
	
		/* 
		 * Set up boot-command area.
		 */
	
		bc = (struct i550bcmd *)I550BOOT;
		bzero((char *)bc, sizeof *bc);
	
		bc->bc_command		= BOOT_START_COMMAND;
		bc->bc_magicWord	= I550_MAGIC_WORD;
		bc->bc_devCount		= I550_DEVICE_COUNT;
		bc->bc_idsCount		= I550_IDS_COUNT;
		bc->bc_destDevId	= I550_DEVICE_ID;
		bc->bc_idsBase		= I550_IDS_BASE;
		bc->bc_idsLength	= I550_IDS_LENGTH;
		bc->bc_srcDevId		= LOCAL_DEVICE_ID;
		bc->bc_status		= I550_STATUS;
		bc->bc_timeout		= I550_TIMEOUT;
		bc->bc_intType		= MB_INT_EDGE;
	
		SELECTOROF(bc->bc_transRqd)	= 0;
		OFFSETOF(bc->bc_transRqd)	= (short)&i550.XmtQ;/* GACK! */
		SELECTOROF(bc->bc_recvRqd)	= 0;
		OFFSETOF(bc->bc_recvRqd)	= (short)&i550.RcvQ;/* ditto */
	
		/* 
		 * The final kick. If this passes, it's there and happy.
		 */

		outbyte(I550PORT, I550_START);
		delay(I550DELAY);	/* "sleep" 2sec */

		if (bc->bc_response == BOOT_SUCCESSFUL_RESULT)
			i550.exists++;	/* it exists! */
	}	

	/* 
	 * Report on outcome.
	 */

	kprintf("iSBC 550 level %x.%x ", (i550fwv/16), (i550fwv%16));
	if (i550.exists == 1)
		kprintf("found\n");
	else{
		kprintf("NOT found, reason 0x%x", i550tstr);
		kprintf(" MIP return 0x%x\n", bc->bc_response);
	}

	return(OK);
}
	