

/* 
 * i550rw.h - Definitions for i550 driver code
 * 
 * Author:	Ralph E. Droms
 * 		Dept. of Computer Sciences
 * 		Purdue University
 * Date:	Fri Sep 21 1984
 * Copyright (c) 1984 Ralph E. Droms
 */

/*
 * $Log$
 */

/*
 * i550.h
 *	Data structures and other definitions for the iSBC 550 Ether
 *	Controller driver.
 *
 * Relevant documents:
 *	1. iSBC 550 Ethernet Communications Controller Programmer's Reference
 *	   Manual, Order Number 121769-001, Intel Corporation.
 *	2. iSBC 550 Ethernet Communications Controller Hardware Reference
 *	   Manual, Order Number 121746-001, Intel Corporation.
 *
 * Chris Kent, Thu Apr 26 08:24:14 1984
 */

/*
 * 550 Driver Parameters.
 */

#define	NSBUF		4		/* Number of "supply" buffers */
#define	NXBUF		1		/* Number of "transmit" buffers */
#define	MAXPACKET	1500		/* this is the max size, if the i550
					*  gets a packet of this size, and
					*  the user has supplied a buffer
					*  that is too small, too bad for
					*  the user
					*/
#define	RETRY_LIMIT	5		/* Max retries to ACK 550 */
#define	RQD_Q_SIZE	4		/* RQD # entries in queue */
#define	LRQE_SIZE	4		/* Log2(sizeof struct mip_rqd) */
#define	I550PORT	0x8a4		/* 550 "wake-up" port */
#define	I550BOOT	0xf690		/* 550 boot command area */
#define	SPL		spl5		/* For Interrupt Mutex */
#define	PI550SLEEP	PRIBIO		/* Sleep priority for 550 */
#define	I550DELAY	60000L		/* argument to delay(): enough for 8Mhz */

#define	ETHER_IPTYPE	0x800		/* EtherType for IP */
#define	ETHER_ARPTYPE	0x806		/* EtherType for ARP */

/* 
 * Macros to make some alignments and other things easier.
 */

typedef	char		plm_ptr[4];		/* PL/M-type "pointer" */
typedef	char		uashort[2];		/* unaligned short */
#define	SHORT(x)	(*((short*)(x)))	/* make believe it's a short */
#define	OFFSETOF(p)	SHORT(p)		/* offset part of "long" pointer */
#define	SELECTOROF(p)	SHORT(p+2)		/* selector part of "long" pointer */
#define	LOWLONG(l)	SHORT(((char *)&l)+2)	/* low 16-bits of value of long */
#define	HIGHLONG(l)	SHORT(((char *)&l)+0)	/* high 16-bits of value of long */
#define	UNSIGNED(c)	((unsigned)((c)&0xff))	/* unsigned extension of char */

/*
 * 550 Boot Presence Command Structure
 */

struct	i550bpres {
	char	bp_command;
	char	bp_response;
	char	bp_testResult;
	char	bp_version;
};

/*  
 * 550 Boot Command Structure
 * Note: manual has in_type and timeout fields reversed.
 */

struct	i550bcmd {
	char	bc_command;
	char	bc_response;
	char	bc_reserved[8];
	short	bc_zeroField;
	short	bc_magicWord;
	char	bc_devCount;
	char	bc_idsCount;
	char	bc_destDevId;
	char	bc_filler;
	char	bc_idsBase;
	char	bc_idsLength;
	char	bc_srcDevId;
	char	bc_status;
	plm_ptr	bc_transRqd;
	plm_ptr	bc_recvRqd;
	char	bc_intType;
	char	bc_timeout;
	short	bc_intAddr;
};

/*
 * 550 Boot Command types
 */

#define	BOOT_PRESENCE_COMMAND	0x01
#define	BOOT_START_COMMAND	0x02
#define	BOOT_SUCCESSFUL_RESULT	0x01
#define	BOOT_BAD_RESULT		0xFF

/*
 * 550 Output Command Types
 */

#define	I550_RESET		0x01
#define	I550_START		0x02
#define	I550_RESET_INTERRUPT	0x04
    
/*
 * 550 Boot and MIP Command Constants
 */

#define	I550_MAGIC_WORD		0x0E08E
#define	I550_DEVICE_COUNT	1
#define	I550_IDS_COUNT		1
#define	LOCAL_DEVICE_ID		0x0000
#define	I550_DEVICE_ID		0x0001
#define	LOCAL_SOCKET_ID		0x0000
#define	I550_IDS_BASE		0x0000
#define	I550_IDS_LENGTH		(16)		/* IDS == All 64K */
#define	I550_STATUS		0x00FF
#define	I550_PORT_ID		0x01
#define	I550_IDS_ID		0x00
#define	I550_TIMEOUT		0xFF		/* No Timeout */
#define	I550_DLO_SIZE		6		/* size of "data-link-object" */
#define	ETHER_MIN_PACKET	46		/* smallest possible packet */
#define	ETHER_ADDR_SIZE		6		/* # bytes in ether address */
#define IP_ADDR_SIZE		4
#define	NO_INTERRUPT		0x00		/* Polling interface */
#define	MB_INT_EDGE		0x03		/* Edge-triggered */
#define MB_INT_LEVEL		0x04		/* Level-triggered */

/*
 * EDL Structure Definitions.
 * Note: ether addresses are high-order-byte in [0] (see ether_frame, below).
 */

struct	edl_hdr	{			/* EDL Header */
	char		eh_reserved[12];
	int		(*eh_processId)();
	char		eh_command;
	char		eh_result;
	short		eh_rspSocket;
};

struct	edl_conn {			/* Connect */
	struct edl_hdr	ec_hdr;
	short		ec_type;
};

struct	edl_disc {			/* Disconnect */
	struct edl_hdr	ed_hdr;
	short		ed_type;
};

struct	edl_amcid {			/* Add Multi-cast ID */
	struct edl_hdr	eam_hdr;
	char		eam_mcid[ETHER_ADDR_SIZE];
};

struct	edl_dmcid {			/* Delete Multi-cast ID */
	struct edl_hdr	edm_hdr;
	char		edm_mcid[ETHER_ADDR_SIZE];
};

struct	edl_sbuf {			/* Supply-Buffer */
	struct edl_hdr	esb_hdr;
	short		esb_length;
	char		esb_dstAddr[ETHER_ADDR_SIZE];
	char		esb_srcAddr[ETHER_ADDR_SIZE];
	short		esb_type;
	char		esb_message[MAXPACKET];
};

struct	edl_xmit {			/* Transmit */
	struct edl_hdr	exm_hdr;
	short		exm_length;
	short		exm_extnsn;
	char		exm_idsId;
	char		exm_ext2;
	short		exm_extLength;
	char		exm_dstAddr[ETHER_ADDR_SIZE];
	char		exm_srcAddr[ETHER_ADDR_SIZE];
	short		exm_type;
	char		exm_message[MAXPACKET];
};

struct	edl_read {			/* Read/ReadC */
	struct edl_hdr	erd_hdr;
	short		erd_dlo;		/* Data-Link Object Desired */
	char		erd_val[I550_DLO_SIZE];	/* Value */
};

/*
 * EDL Command Types
 */

#define	EDL_CONNECT		0x01
#define	EDL_DISCONNECT		0x02
#define	EDL_ADDMCID		0x03
#define	EDL_DELETEMCID		0x04
#define	EDL_TRANSMIT		0x05
#define	EDL_SUPPLYBUF		0x06
#define	EDL_READ		0x08
#define	EDL_READC		0x09
#define GET_XMIT_BUF		0x0a
#define FREE_XMIT_BUF		0x0b

#define	EDL_PACKET_OVHD		14	/* # bytes of non-data in length */

/*
 * EDL READ/READC data-link object request codes.
 */

#define	EDLO_TOTAL_SENT		0
#define	EDLO_PRIMARY_COLLISIONS	1
#define	EDLO_2ND_COLLISIONS	2
#define	EDLO_EXCEED_COLLISIONS	3
#define	EDLO_TRANSMIT_TIMEOUTS	4
#define	EDLO_TOTAL_RECEIVED	5
#define	EDLO_CRC_ERRORS		6
#define	EDLO_FRAMING_ERRORS	7
#define	EDLO_RESOURCE_ERRORS	8
#define	EDLO_HOST_ADDRESS	9
#define	EDLO_LOADING		10
#define	EDLO_MAX_DLO		10

/*
 * EDL returned message types (the last two are locally generated)
 */

#define	EDL_GOOD		0x00
#define	EDL_BAD			0x01
#define	EDL_MESSAGE_NOT_ACKED	0x02
#define	EDL_FRAME_NOT_DELIVERED	0x03
#define	EDL_RESET_FAILED	0x04
#define	EDL_MIP_TYPE_ERROR	0x05

/*
 * MIP Request Queue Entry Structure
 * Note alignment problem on dataptr, datalen.
 */

struct	mip_rqe	{
	char		r_request;
	char		r_sReqId;		/* source request ID */
	char		r_dDevId;		/* destination device ID */
	char		r_dPortId;		/* destination port ID */
	char		r_sDevId;		/* source device ID */
	plm_ptr		r_dataPtr;		/* Pointer to ref'd thing */
	uashort		r_dataLen;		/* length(data) */
	char		r_idsId;		/* IDS ID */
	char		r_oDevId;		/* owner device ID */
	char		r_reserved[3];
};

/*
 * MIP Request Queue Descriptor Structure
 */

struct	mip_rqd	{
	char		d_empty;
	char		d_full;
	char		d_rqSize;
	char		d_rqeLength;
	char		d_giveIndex;
	char		d_giveState;
	char		d_takeIndex;
	char		d_takeState;
	struct mip_rqe	d_rqe[RQD_Q_SIZE];	/* Actual queue entries */
};

/*
 * MIP constants
 */

#define	MIP_Q_NO_LONGER_FULL		0x80
#define	MIP_Q_NO_LONGER_EMPTY		0x01
#define	MIP_NO_CHANGE			0x00
#define	MIP_GIVE_HALT			0x40
#define	MIP_TAKE_HALT			0x40
#define	MIP_GIVE_FACTOR			0x80
#define	MIP_TAKE_FACTOR			0x80

/*
 * MIP Response Types
 */

#define	MIP_SEND_COMMAND		0x70
#define	MIP_MSG_DELIVERED_NO_COPY	0x80
#define	MIP_MSG_DELIVERED_COPY		0x82
#define	MIP_SYSTEM_MEMORY_NAK		0x85
#define	MIP_DEAD_DEVICE			0x89

/*
 * iSBC 550 "device" description.
 */

struct	i550dev	{
	short		d_open;		/* != 0 ==> already open */
	short		d_closed;	/* != 0 ==> has been closed -- ignore received packets */
	char		d_netAddr[ETHER_ADDR_SIZE];	/* this 550's network address */
	short		d_genFBusy;	/* generic frame "busy" */
	struct edl_read	d_genFrame;	/* "generic" frame; big enuf for all */
	struct edl_sbuf	d_sBuf[NSBUF];	/* "supply" buffers */
	short		d_xBufBusy;	/* transmit buffer busy flag */
	struct edl_xmit	d_xBuf;		/* "transmit" buffer (one assumed) */
};

/*
 * Ether-Frame.  Note that all fields have high-order byte in low-order address.
 */

struct	ether_frame {
	char	e_dstAddr[ETHER_ADDR_SIZE];	/* destination addr */
	char	e_srcAddr[ETHER_ADDR_SIZE];	/* source addr */
	short	e_dltc;				/* data-link type-code */
	char	e_data[MAXPACKET];		/* message data */
};

#define	ETHER_MIN_FRAME	(sizeof(struct ether_frame) - MAXPACKET + ETHER_MIN_PACKET)

/* 
 * End of definitions "stolen" from cak/droms experimental code
 * Remaining definitions are Xinu-specific.
 */

#define	I550ONQ		2			/* Return value from enqueue */
#define I550XMITOK	1			/* Return value from tframe */
#define	ERNULL		( (struct i550req *)0)	/* Null request pointer */

/* 
 * i550 request queue.  Read requests are queued up here, pending
 * receipt of sbufs from the controller
 */

#define	MAX_HOST_NAME	14
struct	i550req {
	struct	edl_sbuf *i550sbuf;		/* Pointer to returned sbuf */
	int	i550rpid;			/* Process waiting for an sbuf */
	int	i550rstat;			/* Status of read */
	struct	i550req  *ernext;
};
struct sbuf_qe {		/* linked list of sbufs that are waiting */
    struct edl_sbuf *psbuf;	/* to be read				 */
    struct sbuf_qe *psbuf_qe;
} ;


struct i550 {					/* control block for i550 */
	struct i550dev dev;
	struct mip_rqd XmtQ,
		       RcvQ;

	struct edl_sbuf	*rgpsbuf[NSBUF];	/* queue of packets that */
	int	hd_sbuf,			/* that have arrived but */
		tl_sbuf;			/* have not yet been read */
	int exists;
	int xmitpid;		/* buffer pool id for XMIT buffers	*/
	int read_sem;		/* semaphore for synchronization between*/
				/* i550read and interrupt routines	 */
	struct i550req *reqlst,
		       *reqend;
	char ip_addr[4],
	     sbname[MAX_HOST_NAME],
	     et_addr[ETHER_ADDR_SIZE];
} ;

extern struct i550 i550;



struct	arp_pkt {
	unsigned short	arp_hrd;	/* format of hardware address */
#define ARPHRD_ETHER 	1	/* ethernet hardware address */
	unsigned short	arp_pro;	/* format of proto. address (EPUP_IPTYPE) */
	char	arp_hln;	/* length of hardware address (6) */
	char	arp_pln;	/* length of protocol address (4) */
	unsigned short	arp_op;
#define	ARP_REQUEST	1	/* request to resolve address */
#define	ARP_REPLY	2	/* response to previous request */
	char	arp_sha[6];	/* sender hardware address */
	char	arp_spa[4];	/* sender protocol address */
	char	arp_tha[6];	/* target hardware address */
	char	arp_tpa[4];	/* target protocol address */
};

extern int sys_ready;
extern int mutex_arp;
extern int arp_port;
extern char bcast[];
#define	CXMIT_MAX	3

