/* 
 * Mach Operating System
 * Copyright (c) 1992 Carnegie Mellon University
 * Copyright (c) 1994 Johannes Helander
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 * 
 * CARNEGIE MELLON AND JOHANNES HELANDER ALLOW FREE USE OF THIS
 * SOFTWARE IN ITS "AS IS" CONDITION.  CARNEGIE MELLON AND JOHANNES
 * HELANDER DISCLAIM ANY LIABILITY OF ANY KIND FOR ANY DAMAGES
 * WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 */
/*
 * HISTORY
 * $Log: $
 */
/* 
 *	File: 	serv/ux_server_loop.c
 *	Authors:
 *	Randall Dean, Carnegie Mellon University, 1992.
 *	Johannes Helander, Helsinki University of Technology, 1994.
 *
 *	Server thread utilities and main loop.
 */

#include <serv/import_mach.h>

#include <sys/param.h>
#include <sys/types.h>
#include <sys/proc.h>

mach_port_t ux_server_port_set = MACH_PORT_NULL;

/*
 * Number of server threads available to handle user messages.
 */
struct mutex	ux_server_thread_count_lock = MUTEX_INITIALIZER;
int		ux_server_thread_count = 0;
int		ux_server_thread_min = 4;
int		ux_server_receive_min = 2;
int		ux_server_receive_max = 6;
int		ux_server_thread_max = 80;
int		ux_server_stack_size = 4096*16;
/* This must be at least 5 (wired) + 1 (running) + receive_max */
int		ux_server_max_kernel_threads = 13;

void ux_create_single_server_thread(); /* forward */

void
ux_server_init()
{
	mach_port_t first_port;
	ur_cthread_t tmp;
	kern_return_t kr;

	kr = mach_port_allocate(mach_task_self(), MACH_PORT_RIGHT_PORT_SET,
				&ux_server_port_set);

	cthread_set_kernel_limit(ux_server_max_kernel_threads);

}

void
ux_server_add_port(port)
	mach_port_t	port;
{
	(void) mach_port_move_member(mach_task_self(),
				     port, ux_server_port_set);
}

void
ux_server_remove_port(port)
	mach_port_t	port;
{
	(void) mach_port_move_member(mach_task_self(), port, MACH_PORT_NULL);
}

void *ux_thread_bootstrap (cthread_fn_t real_routine)
{
	proc_invocation_t pk;
	any_t ret;
	struct proc_invocation pkdata;

	pk = &pkdata;
	if (pk == NULL)
	    panic("ux_thread_bootstrap");
	queue_init(&pk->k_servers_chain);
	pk->k_wchan = NULL;
	pk->k_wmesg = NULL;
	/* reset k_p later if system proc or something else */
	pk->k_p = 0;
	pk->k_master_lock = 0;
	pk->k_reply_msg = NULL;
	pk->k_current_size = 0;
	pk->k_ipl = 0;
	pk->k_timedout = FALSE;
	pk->k_flag = 0;
	pk->cthread = cthread_self(); /* to make debugging easier */
	mutex_init(&pk->lock);
	pk->event = 0;
	pk->condition = 0;

	cthread_set_ldata(cthread_self(), pk);

	ret = ((*real_routine)((any_t)0));

	if (pk->k_wchan || pk->k_p != 0 || pk->k_master_lock
	    || pk->k_ipl || pk->k_reply_msg)
	{
		panic("ux_thread_bootstrap");
	}
	return ret;
}

/*
 * Create a thread
 */
void
ux_create_thread(routine)
	cthread_fn_t	routine;
{
	cthread_detach(cthread_fork(ux_thread_bootstrap,
				    (void *) routine));
}

any_t	ux_server_loop();	/* forward */

void
ux_create_server_thread()
{
	ux_create_thread(ux_server_loop);
}

void
ux_server_thread_busy()
{
	cthread_msg_busy(ux_server_port_set,
			 ux_server_receive_min,
			 ux_server_receive_max);
	mutex_lock(&ux_server_thread_count_lock);
	if (--ux_server_thread_count < ux_server_thread_min) {
	    mutex_unlock(&ux_server_thread_count_lock);
	    ux_create_server_thread();
	}
	else {
	    mutex_unlock(&ux_server_thread_count_lock);
	}
}

void
ux_server_thread_active()
{
	cthread_msg_active(ux_server_port_set,
			 ux_server_receive_min,
			 ux_server_receive_max);
	mutex_lock(&ux_server_thread_count_lock);
	++ux_server_thread_count;
	mutex_unlock(&ux_server_thread_count_lock);
}

#define	ux_server_thread_check() \
	(ux_server_thread_count > ux_server_thread_max)

/*
 * Main loop of server.
 */
/*ARGSUSED*/
any_t
ux_server_loop(arg)
	any_t	arg;
{
	register kern_return_t	ret;
	proc_invocation_t pk = get_proc_invocation();

	union request_msg {
	    mach_msg_header_t	hdr;
	    mig_reply_header_t	death_pill;
	    char		space[8192];
	} msg_buffer_1, msg_buffer_2;

	mach_msg_header_t * request_ptr;
	mig_reply_header_t * reply_ptr;
	mach_msg_header_t * tmp;

	char	name[64];

	sprintf(name, "ST 0x%x", cthread_self());
	cthread_set_name(cthread_self(), name);

	pk->k_p = NULL;		/* fix device reply instead XXX */
	ux_server_thread_active();

	request_ptr = &msg_buffer_1.hdr;
	reply_ptr = &msg_buffer_2.death_pill;

	do {
	    ret = cthread_mach_msg(request_ptr, MACH_RCV_MSG,
				   0, sizeof msg_buffer_1, ux_server_port_set,
				   MACH_MSG_TIMEOUT_NONE, MACH_PORT_NULL,
				   ux_server_receive_min,
				   ux_server_receive_max);
	    if (ret != MACH_MSG_SUCCESS)
		panic("ux_server_loop: receive", ret);
	    while (ret == MACH_MSG_SUCCESS) {
		if (!seqnos_memory_object_server(request_ptr,&reply_ptr->Head))
		if (!bsd_1_server(request_ptr, &reply_ptr->Head))
		if (!ux_generic_server(request_ptr, &reply_ptr->Head))
		if (!exc_server(request_ptr, &reply_ptr->Head))
		if (!seqnos_notify_server(request_ptr, &reply_ptr->Head))
		    bad_request_server(request_ptr, &reply_ptr->Head);

		/* Don't lose a sequence number if a type check failed */
		if (reply_ptr->RetCode == MIG_BAD_ARGUMENTS) {
			printf("MiG type check failed: req id %d port=x%x\n",
			       request_ptr->msgh_id,
			       request_ptr->msgh_local_port);
			bad_request_server(request_ptr, &reply_ptr->Head);
		}

		if (reply_ptr->Head.msgh_remote_port == MACH_PORT_NULL) {
		    /* no reply port, just get another request */
		    break;
		}

		if (reply_ptr->RetCode == MIG_NO_REPLY) {
		    /* deallocate reply port right */
		    (void) mach_port_deallocate(mach_task_self(),
					reply_ptr->Head.msgh_remote_port);
		    break;
		}

		ret = cthread_mach_msg(&reply_ptr->Head,
				       MACH_SEND_MSG|MACH_RCV_MSG,
				       reply_ptr->Head.msgh_size,
				       sizeof msg_buffer_2,
				       ux_server_port_set,
				       MACH_MSG_TIMEOUT_NONE,
				       MACH_PORT_NULL,
				       ux_server_receive_min,
				       ux_server_receive_max);
		if (ret != MACH_MSG_SUCCESS) {
		    if (ret == MACH_SEND_INVALID_DEST) {
			/* deallocate reply port right */
			/* XXX should destroy entire reply msg */
			(void) mach_port_deallocate(mach_task_self(),
					reply_ptr->Head.msgh_remote_port);
		    } else {
			    panic("ux_server_loop: rpc x%x", ret);
		    }
		}

		tmp = request_ptr;
		request_ptr = (mach_msg_header_t *) reply_ptr;
		reply_ptr = (mig_reply_header_t *) tmp;
	    }

	} while (!ux_server_thread_check());

	ux_server_thread_busy();

        printf("Server loop done: %s\n",name);

	return ((any_t)0);	/* exit */
}

