/* 
 * Copyright (c) 1992, 1994, The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 * All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software is hereby
 * granted provided that (1) source code retains these copyright, permission,
 * and disclaimer notices, and (2) redistributions including binaries
 * reproduce the notices in supporting documentation, and (3) all advertising
 * materials mentioning features or use of this software display the following
 * acknowledgement: ``This product includes software developed by the
 * Computer Systems Laboratory at the University of Utah.''
 *
 * THE UNIVERSITY OF UTAH AND CSL ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSL DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSL requests users of this software to return to csl-dist@cs.utah.edu any
 * improvements that they make and grant CSL redistribution rights.
 *
 * 	Utah $Hdr: vm_machdep.c 1.4 94/12/16$
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/proc.h>
#include <sys/malloc.h>
#include <sys/buf.h>
#include <sys/vnode.h>
#include <sys/user.h>
#include <sys/assert.h>
#include <sys/som.h>
#include <machine/cpu.h>

#include <vm/vm.h>

/*
 * Have to make sure the U-area we dump looks just like the Utah BSD kind.
 */
struct	bsd_pcb {
	char	pad0[0x28c-0x000];
	struct	bsd_exechdr pcb_exec;	/* BSD exec header */
	char	pad1[0x2e4-0x2c0];
	u_int	pcb_pgsz;		/* machine page size (bytes) */
	char	pad2[0x2f8-0x2e8];
};

struct	bsd_user {
	struct	bsd_pcb bsdu_pcb;
	char	pad1[0x3e8-0x2f8];
	size_t	bsdu_tsize;		/* text size (clicks) */
	size_t	bsdu_dsize;		/* data size (clicks) */
	size_t	bsdu_ssize;		/* stack size (clicks) */
};

/*
 *	fake_bsd_u:
 *
 *	Fake an BSD u-area structure for the specified thread.
 *	Only "interesting" fields are filled in.
 */
void
fake_bsd_u(bup, p, thread)
	register struct bsd_user *bup;
	struct proc		 *p;
	register thread_t	 thread;
{
	bzero((caddr_t) bup, sizeof(struct bsd_user));
	
	bup->bsdu_pcb.pcb_pgsz = NBPG;

#if notyet
	bup->bsdu_pcb.pcb_exec.a_tmem = (int)p->p_utask.uu_text_start;
	bup->bsdu_pcb.pcb_exec.a_dmem = (int)p->p_utask.uu_data_start;

	bup->bsdu_tsize = p->p_utask.uu_tsize;
	bup->bsdu_dsize = p->p_utask.uu_dsize;
	bup->bsdu_ssize = p->p_utask.uu_ssize;
#endif
}

/*
 * Dump the machine specific header information at the start of a core dump.
 */
mach_error_t
cpu_coredump(p, vp, cred)
	struct proc *p;
	struct vnode *vp;
	struct ucred *cred;
{
#if notyet
	vm_offset_t	 		stack_addr;
	vm_size_t	 		stack_size;
	int		 		error, offset;
	struct parisc_thread_state 	ts;
	unsigned int 			count;
	struct bsd_user			fake_uarea;

	/*
	 * Get the registers for the thread.
	 */
	count = PARISC_THREAD_STATE_COUNT;
	(void) thread_get_state(p->p_thread, PARISC_THREAD_STATE,
				(thread_state_t)&ts,
				&count);

	/*
	 * Fake the stack segment size from the current
	 * stack pointer value.
	 */
	stack_addr = USRSTACK;
	stack_size = round_page(ts.r30 - stack_addr + 4);
	p->p_utask.uu_ssize = btoc(stack_size);

	/*
	 * Dump a BSD style U-area (base of kernel stack).
	 */
	fake_bsd_u(&fake_uarea, p, p->p_thread);
	error = vn_rdwr(UIO_WRITE, vp, (caddr_t)&fake_uarea, sizeof fake_uarea,
			(off_t)0, UIO_SYSSPACE, IO_NODELOCKED|IO_UNIT,
			cred, (int *)0, p);
	if (error)
		return (error);

	/*
	 * Dump saved state info (top of kernel stack). 
	 */
	offset = ctob(1),		/* XXX BSD offsets */
	error = vn_rdwr(UIO_WRITE, vp, (caddr_t)&ts, sizeof ts, (off_t)offset,
			UIO_SYSSPACE, IO_NODELOCKED|IO_UNIT, cred, (int *)0, p);
	if (error)
		return (error);

	/*
	 * Write the data segment.
	 */
	offset += ctob(2);		/* XXX BSD offsets */
	error = vn_rdwr(UIO_WRITE, vp, (caddr_t)u.u_data_start,
			(int)ctob(u.u_dsize), (off_t)offset,
			UIO_SYSSPACE, IO_NODELOCKED|IO_UNIT, cred, (int *)0, p);
	if (error)
		return (error);

	/*
	 * Write the stack.
	 */
	offset += ctob(u.u_dsize);
	error = vn_rdwr(UIO_WRITE, vp, (caddr_t)stack_addr,
			(int)stack_size, (off_t)offset,
			UIO_SYSSPACE, IO_NODELOCKED|IO_UNIT, cred, (int *)0, p);
	return (error);
#else
	return EIO;
#endif
}

/*
 * Move pages from one kernel virtual address to another.
 * Both addresses are assumed to reside in the Sysmap,
 * and size must be a multiple of CLSIZE.
 */
void
pagemove(from, to, size)
	register caddr_t from, to;
	int size;
{
#ifdef LITES
	kern_return_t kr;

	kr = vm_write(mach_task_self(), (vm_offset_t) to,
		      (vm_offset_t) from, size);
	assert(kr == KERN_SUCCESS);
#if 0
	/* The mapping should stay. So the old junk stays around ... */
	/* There must be better ways to handle this whole thing on Mach... */
	kr = vm_deallocate(mach_task_self(), from, size);
	assert(kr == KERN_SUCCESS);
#endif
#else
	register vm_offset_t pa;

	while (size > 0) {
		pa = pmap_extract(kernel_pmap, (vm_offset_t)from);
		pmap_remove(kernel_pmap,
			    (vm_offset_t)from, (vm_offset_t)from + PAGE_SIZE);
		pmap_enter(kernel_pmap,
			   (vm_offset_t)to, pa, VM_PROT_READ|VM_PROT_WRITE, 1);
		from += PAGE_SIZE;
		to += PAGE_SIZE;
		size -= PAGE_SIZE;
	}
#endif
}
