/* 
 * Mach Operating System
 * Copyright (c) 1994 Johannes Helander
 * Copyright (c) 1994, The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 * 
 * JOHANNES HELANDER AND THE UNIVERSITY OF UTAH AND CSL ALLOWS FREE
 * USE OF THIS SOFTWARE IN ITS "AS IS" CONDITION.  JOHANNES HELANDER
 * AND THE UNIVERSITY OF UTAH AND CSL DISCLAIMS ANY LIABILITY OF ANY
 * KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS
 * SOFTWARE.
 *
 * 	Utah $Hdr: e_machinedep.c 1.4 94/12/16$
 */

#include <e_defs.h>
#include <syscall_table.h>
#include <parisc/eregs.h>

char *atbin_names[] =
{
	ATSYS_NAMES("parisc_")
};

/*
 * For fork/vfork, the parent returns 0 in rval[1] (ret1), child returns 1.
 *
 * XXX not machine or HPBSD specific, should be elsewhere.  e_bsd.c?
 */
errno_t e_machine_fork(
	int			*argp,
	int			*rval,
	struct emul_regs	*regs)
{
	errno_t err;

	err = e_fork((pid_t *)rval);
	if (err)
		return err;
	if (rval[0])
		rval[1] = 0;
	else
		rval[1] = 1;

	return ESUCCESS;
}

errno_t e_machine_vfork(
	int			*argp,
	int			*rval,
	struct emul_regs	*regs)
{
	errno_t err;

	err = e_vfork((pid_t *)rval);
	if (err)
		return err;
	if (rval[0])
		rval[1] = 0;
	else
		rval[1] = 1;

	return ESUCCESS;
}

errno_t
e_fork_call(boolean_t isvfork, pid_t *pid, boolean_t *ischild)
{
	struct parisc_thread_state state;
	int intr, rv[2];
	errno_t error;
	volatile int x;

	/*
	 * Set up registers for child.  It resumes on its own stack.
	 */

	bzero((caddr_t) &state, sizeof state);
	x = emul_save_state(&state);

	if (x != 0) {
		/*
		 * XXX need to pass in regs so we can set a flag telling
		 * emul_trampoline to load child sr0 value from sr4.
		 * Right now the trampoline code just uses sr4 for its
		 * return space which works fine as long as there is
		 * only one space per process.
		 */
		*ischild = TRUE;
		*pid = x;
		child_init();
		if (syscall_debug > 1)
			e_emulator_error("CHILD parent=%d", x);
		return 0;
	}

	/*
	 * Arrange for a non-zero return in the child
	 * for emul_save_state above.
	 */
	state.r28 = 1;
	
	/*
	 * Create the child.
	 */
	if (isvfork)
		error = bsd_vfork(our_bsd_server_port, &intr,
				  (int *)&state, PARISC_THREAD_STATE_COUNT, rv);
	else
		error = bsd_fork(our_bsd_server_port, &intr,
				 (int *)&state, PARISC_THREAD_STATE_COUNT, rv);
	
	if (error == 0)
		*pid = rv[0];

	*ischild = FALSE;
	return error;
}

/*
 * XXX make sure kframe info is right
 */
noreturn emul_exec_start(
	int			*kframe,
	enum binary_type	binary_type,
	int			argc,
	char			*argv[],
	char			*envp[],
	struct exec_load_info	*li)
{
	struct parisc_thread_state state;

	bzero((caddr_t) &state, sizeof state);
	(void) emul_save_state(&state);

	/*
	 * Load argument registers:
	 *	arg0 = argc
	 *	arg1 = argv
	 *	arg2 = envp
	 *	arg3 = PA-RISC version * 10 (XXX always 1.1 for now)
	 * Note that DP is set by crt0 included in all binaries.
	 */
	state.r26 = argc;
	state.r25 = (int) argv;
	state.r24 = (int) envp;
	state.r23 = 11;
	state.iioq_head = (unsigned) li->pc | 3;
	state.iioq_tail = state.iioq_head + 4;

	if (syscall_debug > 1)
		e_emulator_error("emul_exec_start: x%x(x%x, x%x, x%x, x%x)\n",
				 state.iioq_head,
				 state.r26, state.r25, state.r24, state.r23);
	emul_load_state(&state);
	/*NOTREACHED*/
}

/*
 * PA-specific entry point for HPBSD wait/wait3.  Input formats:
 *	wait(status_p, xxx, xxx, 0xFacade)
 *	wait3(status_p, options, rusage_p, 0xFacade3)
 * Both convert to wait4 format:
 *	wait4(pid, status_p, options, rusage_p)
 */
errno_t
e_machine_wait(
	int			*argp,
	int			*rval,
	struct emul_regs	*regs)
{
	errno_t err;

#define WAIT_ANY (-1)

	if (argp[3] == 0xFacade3)
		err = e_hpbsd_wait4(WAIT_ANY, argp[0], argp[1], argp[2], rval);
	else
		err = e_hpbsd_wait4(WAIT_ANY, argp[0], 0, 0, rval);

	return err;
}

/* 
 * Process signals after a system call.
 * Called emul_syscall if needed.
 *
 * XXX This signal code does not care about sigstacks and similar
 * XXX messy stuff. However, it works in most simple cases.
 */
void take_signals_and_do_sigreturn(struct emul_regs *regs)
{
	struct parisc_thread_state state;
	volatile int x;

	bzero((caddr_t) &state, sizeof state);
	x = emul_save_state(&state);

	if (x != 0) {
		emul_set_dp(emul_dp());
		return;
	}

	/* XXX should we set state.flags to SS_INSYSCALL here? */

	/* XXX make sure state reflects user's DP, not emulator's */
	state.r27 = regs->dp;

	while (take_a_signal(&state))
	    ;

	if (state.r2 == 0) {
		e_emulator_error ("would jump to zero!\n");
		return;
	}
	/*
	 * Now return where requested (or from this function if state unchanged)
	 */
	if (state.r28 == 0 &&
	    state.iioq_head > EMULATOR_BASE && state.iioq_head < EMULATOR_END)
		state.r28 = 1;		/* make sure we return above */
	emul_load_state(&state);
	/*NOTREACHED*/
}

/*
 * Take a signal.
 * Called both after a system call detected signal and asynchronous signals
 * delivered via the server (to the emulator signal thread).
 */
boolean_t take_a_signal(struct parisc_thread_state *state)
{
	int old_mask, old_onstack, sig, code, new_sp, cmu_binary;
	boolean_t interrupt;
	int error, edp;
	void (*handler)();
	struct sigcontext sc;

	/*
	 * Get the signal to take from the server.  It also
	 * switches the signal mask and the stack, so we must -- XXX Not stack
	 * be off the old user stack before calling it.
	 */
	error = bsd_take_signal(our_bsd_server_port,
				&interrupt,
				&old_mask,
				&old_onstack,
				&sig,
				&code,
				(int *) &handler,
				&new_sp,
				&cmu_binary);

	/*
	 * If there really were no signals to take, return.
	 */
	if (error || sig == 0)
		return FALSE;

	if (syscall_debug > 1)
		e_emulator_error("SIG%d x%x", sig, handler);

	/*
	 * Put the signal context and signal frame on the signal stack.
	 */
	if (new_sp != 0) {
		e_emulator_error("take_signals: stack switch needed");
		/*
		 * Build signal frame and context on user's stack.
		 */
	}
	/*
	 * Build the signal context to be used by sigreturn.
	 * Both BSD and HP-UX use the same format.
	 *
	 * XXX actually not true because of HP-UX shared libraries,
	 *     the handler address may not be an actual address.
	 *     Worry about this later.
	 */
	sc.sc_onstack = old_onstack;
	sc.sc_mask = old_mask;
		
	sc.sc_sp = state->r30;
	sc.sc_fp = 0;
	sc.sc_ap = (int) state;	/* XXX not quite right */
	sc.sc_ps = state->cr22;

	/*
	 * XXX need to think about the INTRAP/INSYSCALL distinction
	 *     and whether we need to use PC queues vs. sr0/r31.
	 */
	sc.sc_pcsqh = sc.sc_pcsqt = state->sr4;
	sc.sc_pcoqh = state->iioq_head;
	sc.sc_pcoqt = state->iioq_tail;
		
	/*
	 * Then just call it
	 * XXX make sure we call with the user's DP, not the emulator's
	 */
	emul_set_dp(state->r27);
	(*handler)(sig, code, &sc);
	emul_set_dp(emul_dp());
		
	state->r30 = sc.sc_sp;
	state->cr22 = sc.sc_ps;
	
	/*
	 * XXX need to think about the INTRAP/INSYSCALL distinction
	 *     and whether we need to use PC queues vs. sr0/r31.
	 */
	state->iioq_head = sc.sc_pcoqh;
	state->iioq_tail = sc.sc_pcoqt;

	if (syscall_debug > 2)
		e_emulator_error("return_SIG%d", sig);

	/* Re-enable signal */
	error = bsd_sigreturn(our_bsd_server_port,
			      &interrupt,
			      old_onstack & 01,
			      old_mask, &cmu_binary);
	return TRUE;
}

/*
 * Return from a signal.
 * May be called from a signal handler (sc_ap != 0)
 * or from a longjmp (sc_ap == 0) not related to a signal.
 */
errno_t e_machine_sigreturn(
	int	  		*argp,
	int			*rvals,
	struct emul_regs	*regs)
{
	struct sigcontext *sc;
	struct parisc_thread_state state;
	int interrupt, cmu_binary; /* XXX */

	sc = (struct sigcontext *) argp[0];
	if (sc->sc_ap) {
		/*
		 * Called from a signal handler, complete state is
		 * contained in a thread_state structure pointed to
		 * by sc_ap.
		 */
		state = *(struct parisc_thread_state *)sc->sc_ap;
	} else {
		/* longjmp */
		/* Re-enable signal */
		bsd_sigreturn(our_bsd_server_port,
			      &interrupt,
			      sc->sc_onstack & 01,
			      sc->sc_mask, &cmu_binary);

		/* Arrange to return to the caller of setjmp.  */
		regs->reto = (int *) sc->sc_pcoqh;

		/* Arrange to reset the stack pointer.  */
		regs->flags |= SB_NEW_SP_IN_RET1;
		regs->ret1 = sc->sc_sp;

		/* Return value from longjmp.  */
		return sc->sc_pcoqt;
	}

	state.r30 = sc->sc_sp;
	state.cr22 = sc->sc_ps;

	/*
	 * XXX need to think about the INTRAP/INSYSCALL distinction
	 *     and whether we need to use PC queues vs. sr0/r31.
	 */
	state.iioq_head = sc->sc_pcoqh;
	state.iioq_tail = sc->sc_pcoqt;

	if (syscall_debug > 2)
		e_emulator_error("bsd_SIGRETURN");

	/* Re-enable signal */
	bsd_sigreturn(our_bsd_server_port,
		      &interrupt,
		      sc->sc_onstack & 01,
		      sc->sc_mask, &cmu_binary);

	/*
	 * Now return where requested.
	 * If that is back into take_signals_and_do_sigreturn (handler
	 * called sigreturn itself) set ret0 so we don't loop.
	 */
	if (state.r28 == 0 &&
	    state.iioq_head > EMULATOR_BASE && state.iioq_head < EMULATOR_END)
		state.r28 = 1;
	emul_load_state(&state);
	/*NOTREACHED*/

	emul_panic("e_bsd_sigreturn: emul_load_state returned!");
}
