/*	$NetBSD: net_write.c,v 1.2.22.1 2023/08/11 13:40:02 martin Exp $	*/

/*
 * Copyright (c) 1995, 1996, 1997, 1998 Kungliga Tekniska Högskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <config.h>

#include <krb5/roken.h>

/*
 * Like write but blocking sockets never return partial data, i.e. we retry on
 * EINTR.  With non-blocking sockets (EWOULDBLOCK or EAGAIN) we return the
 * number of bytes written.
 */

#ifndef _WIN32

ROKEN_LIB_FUNCTION ssize_t ROKEN_LIB_CALL
net_write (rk_socket_t fd, const void *buf, size_t nbytes)
{
    const char *cbuf = (const char *)buf;
    ssize_t count;
    size_t rem = nbytes;

    while (rem > 0) {
	count = write (fd, cbuf, rem);
	if (count < 0) {
            switch (errno) {
            case EINTR:
		continue;
#if defined(EAGAIN) && EAGAIN != EWOULDBLOCK
            case EAGAIN:
#endif
            case EWOULDBLOCK:
                return nbytes - rem;
            default:
		return count;
            }
	}
	cbuf += count;
	rem -= count;
    }
    return nbytes;
}

#else

ROKEN_LIB_FUNCTION ssize_t ROKEN_LIB_CALL
net_write(rk_socket_t sock, const void *buf, size_t nbytes)
{
    const char *cbuf = (const char *)buf;
    ssize_t count;
    size_t rem = nbytes;
#ifdef SOCKET_IS_NOT_AN_FD
    int use_write = 0;
#endif

    while (rem > 0) {
#ifdef SOCKET_IS_NOT_AN_FD
	if (use_write)
	    count = _write (sock, cbuf, rem);
	else
	    count = send (sock, cbuf, rem, 0);

	if (use_write == 0 &&
	    rk_IS_SOCKET_ERROR(count) &&
	    (rk_SOCK_ERRNO == WSANOTINITIALISED ||
             rk_SOCK_ERRNO == WSAENOTSOCK)) {
	    use_write = 1;

	    count = _write (sock, cbuf, rem);
	}
#else
	count = send (sock, cbuf, rem, 0);
#endif
	if (count < 0) {
            if (!use_write) {
                switch (rk_SOCK_ERRNO) {
                case WSAEINTR:
                    continue;
                case WSAEWOULDBLOCK:
                    return nbytes - rem;
                default:
                    return count;
                }
            } else {
                switch (errno) {
                case EINTR:
                    continue;
                case EWOULDBLOCK:
                    return nbytes - rem;
                default:
                    return count;
                }
            }
	}
	cbuf += count;
	rem -= count;
    }
    return nbytes;
}

#endif
