/* File:	MagnifyView.m - View class for 'Magnify'
 *
 * By:		Christopher Lane
 *		Symbolic Systems Resources Group
 *		Knowledge Systems Laboratory
 *		Stanford University
 *
 * Date:	6 March 1990
 *
 * Copyright:	1990 by The Leland Stanford Junior University.  This program
 *		may be distributed without restriction for non-commercial use.
 */

#import "Magnify.h"
#import "MagnifyView.h"

#define NX_NOBUTTONS (0)

#define NXSetWindowLevel _NXSetWindowLevel
extern int NXSetWindowLevel(int, int);

@implementation MagnifyView

+ newFrame:(const NXRect *) frameRect;
{
	self = [super newFrame:frameRect];
	lock = mutex_alloc();
	frozen = NO;
	mouse = frameRect->origin;
	
	[self createWindows:[NXApp getScale]];
	
	return self;
}

- (BOOL) acceptsFirstResponder { return YES; }

- copy:sender
{
	char *buffer;
	NXStream *stream;
	int length, maxLength;
	Pasteboard *pasteboard = [NXApp pasteboard];
	
	[NXWait push];

	[pasteboard declareTypes:&NXPostScriptPboard num:1 owner:self];

	stream = NXOpenMemory(NULL, 0, NX_WRITEONLY);
	[self copyPSCodeInside:&bounds to:stream];
	NXFlush(stream);

	NXGetMemoryBuffer(stream, &buffer, &length, &maxLength);
	[pasteboard writeType:NXPostScriptPboard data:buffer length:length];
	NXCloseMemory(stream, NX_FREEBUFFER);
	
	[NXWait pop];
	
	return self;
}

- drawSelf:(const NXRect *) rects :(int) rectCount
{
	mutex_lock(lock);
	NXImageBitmap(&virtualBounds, width, height, bps, spp, NX_PLANAR, NX_MONOTONICMASK, (void *) data, NULL, NULL, NULL, NULL);
	mutex_unlock(lock);

	return self;
}

- free
{
	if (invisibleWindow != nil) [invisibleWindow close];
	if (bitmap != nil) {
		[[bitmap window] close];
		[bitmap free];
		}
	if (data != NULL) free((void *) data);
	
	return [super free];
}

- sizeTo:(NXCoord) newWidth :(NXCoord) newHeight;
{	
	id result;
	
	mutex_lock(lock);
	result = [super sizeTo:newWidth :newHeight];
	mutex_unlock(lock);
	
	[self createWindows:[NXApp getScale]];
	
	return result;
}

- mouseMoved:(NXEvent *)theEvent
{
	if (!mutex_try_lock(lock)) return [nextResponder mouseMoved:theEvent];
	while([NXApp peekAndGetNextEvent:NX_MOUSEMOVEDMASK] != NULL);
	[window removeFromEventMask:NX_MOUSEMOVEDMASK];

	mouse = theEvent->location;
	[window convertBaseToScreen:&mouse];

	mutex_unlock(lock);
	
	[[self updateBitmap:&mouse] display];
	
	[window addToEventMask:NX_MOUSEMOVEDMASK];

	return [nextResponder mouseMoved:theEvent];
}

- updateBitmap:(NXPoint *) point
{
	NXPoint origin = { 0.0, 0.0 };
	NXSize size;

	mutex_lock(lock);

	[bitmap getSize:&size];
	[[invisibleWindow moveTo:point->x - (size.width / 2) :point->y - (size.height / 2)] orderFront:self];
	
	[bitmap lockFocus];
	[bitmap composite:NX_CLEAR toPoint:&origin];
	PScomposite(origin.x, origin.y, size.width, size.height, [invisibleWindow gState], origin.x, origin.y, NX_COPY);
	[bitmap unlockFocus];
	
	[invisibleWindow orderOut:self];
	
	[bitmap readImage:(void *) data];
	
	mutex_unlock(lock);

	return self;
}

- (BOOL) isFrozen { return frozen; }

- toggleFrozen:sender
{
	mutex_lock(lock);
	if (frozen = !frozen) [window removeFromEventMask:NX_MOUSEMOVEDMASK];
	else [window addToEventMask:NX_MOUSEMOVEDMASK];
	mutex_unlock(lock);
	
	[[NXApp mainMenu] update];
	
	return self;
}

- createWindows:(float) scale;
{
	Window *bitmapWindow;
	NXRect rect;
	int bytes;
	
	mutex_lock(lock);
	
	rect = bounds;
	rect.size.width = ceil(bounds.size.width / scale);
	rect.size.height = ceil(bounds.size.height / scale);

	virtualBounds = rect;
	virtualBounds.size.width *= scale;
	virtualBounds.size.height *= scale;
	
	if (invisibleWindow != nil) [invisibleWindow sizeWindow:rect.size.width :rect.size.height];
	else {
		invisibleWindow = [Window newContent:&rect style:NX_PLAINSTYLE backing:NX_NONRETAINED buttonMask:NX_NOBUTTONS defer:NO];
		PSsetautofill(NO, [invisibleWindow windowNum]);
		NXSetWindowLevel([invisibleWindow windowNum], NX_MAINMENULEVEL);
		}

	if (bitmap != nil) {
		[[bitmap window] close];
		[bitmap free];
		}
	bitmapWindow = [Window newContent:&rect style:NX_PLAINSTYLE backing:NX_RETAINED buttonMask:NX_NOBUTTONS defer:NO];
	[bitmapWindow setFreeWhenClosed:NO];
	bitmap = [Bitmap newRect:&rect type:NX_UNIQUEBITMAP window:bitmapWindow];
	
	[bitmap lockFocus];
	[bitmap imageSize:&bytes width:&width height:&height bps:&bps spp:&spp inRect:NULL];
	[bitmap unlockFocus];

	data = (char *) ((data == NULL) ? malloc((size_t) bytes) : realloc((void *) data, (size_t) bytes));
	
	mutex_unlock(lock);

	[[self updateBitmap:&mouse] display];

	return self;
}

@end
