/*
**  wt -- a 3d game engine
**
**  Copyright (C) 1994 by Chris Laurel
**  email:  claurel@mr.net
**  snail mail:  Chris Laurel, 5700 W Lake St #208,  St. Louis Park, MN  55416
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <stdio.h>
#include <string.h>
#include "wt.h"
#include "error.h"
#include "wtmem.h"
#include "color.h"
#include "graphfile.h"


/* MAGIC_LENGTH must be the length of the longest MAGIC string. */
#define MAGIC_LENGTH 5
#define GIF_MAGIC "GIF87"
#define GIF89_MAGIC "GIF89"
#define PPM_MAGIC "P6"


extern Graphic_file *LoadPPM(FILE *fp, char *filename);
extern Graphic_file *LoadGIF(FILE *fp, char *filename);

static Graphic_file_format check_format(FILE *fp, char *filename);


Graphic_file *new_graphic_file()
{
     Graphic_file *gf = wtmalloc(sizeof(Graphic_file));

     gf->transparent_entry = -1;

     return gf;
}


void free_graphic_file(Graphic_file *gfile)
{
     if (gfile != NULL) {
	  if (gfile->palette != NULL)
	       wtfree(gfile->palette);
	  if (gfile->bitmap != NULL)
	       wtfree(gfile->bitmap);
     }
     wtfree(gfile);
}


RGBcolor graphic_file_pixel(Graphic_file *gfile, int x, int y)
{
     if (gfile->type == gfPaletted) {
	  int index;

	  index = gfile->bitmap[y * gfile->width + x];
	  return gfile->palette[index];
     } else {
	  unsigned char *pixel = gfile->bitmap + (y * gfile->width + x) * 3;
	  RGBcolor rgb;

	  rgb.red = *pixel++;
	  rgb.green = *pixel++;
	  rgb.blue = *pixel;

	  return rgb;
     }
}

	  
Graphic_file *read_graphic_file(char *filename)
{
     FILE *fp;
     Graphic_file_format format;
     Graphic_file *gfile = NULL;


     if ((fp = fopen(filename, "rb")) == NULL)
	  fatal_error("Could not open texture %s", filename);

     format = check_format(fp, filename);
     fseek(fp, 0, SEEK_SET);

     switch (format) {
	case formatGIF89:
	case formatGIF87:
	  gfile = LoadGIF(fp, filename);
	  break;

	case formatPPM:
	  gfile = LoadPPM(fp, filename);
	  break;

	case formatUnknown:
	  fatal_error("Unknown graphic file format.\n");
	  break;

	default:
	  fatal_error("The graphic file reading code is really broken.\n");
	  break;
     }

     if (gfile == NULL)
	  fatal_error("Error reading texture %s\n");

     fclose(fp);

     return gfile;
}
	  

static Graphic_file_format check_format(FILE *fp, char *filename)
{
     char magic[MAGIC_LENGTH];

     if (fread(magic, 1, MAGIC_LENGTH, fp) != MAGIC_LENGTH)
	  fatal_error("Error reading texture %s.", filename);

     if (strncmp(magic, PPM_MAGIC, sizeof(PPM_MAGIC) -  1) == 0)
	  return formatPPM;
     else if (strncmp(magic, GIF_MAGIC, sizeof(GIF_MAGIC) - 1) == 0)
	  return formatGIF87;
     else if (strncmp(magic, GIF89_MAGIC, sizeof(GIF89_MAGIC) - 1) == 0)
	  return formatGIF89;
     else
	  return formatUnknown;
}
